package api

import (
	"encoding/json"
	"fmt"

	"net/http"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend"
	"github.com/jixwanwang/apiutils"
)

func (T *API) updateProperties(w http.ResponseWriter, r *http.Request) {
	ctx := chitin.Context(w, r)

	decoder := json.NewDecoder(r.Body)
	var params map[string]interface{}
	err := decoder.Decode(&params)

	if _, ok := params["name"]; !ok {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Missing param: name"))
		return
	}

	if _, ok := params["properties"]; !ok {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Missing param: properties"))
		return
	}

	// string pointer to capture null values
	properties, ok := params["properties"].(map[string]interface{})

	if !ok {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, fmt.Sprintf("malformed properties param")))
		return
	}

	// Break out separate maps for updating and deleting
	toUpdate := make(map[string]string)
	toDelete := make(map[string]string)
	for k, v := range properties {
		if v != nil {
			toUpdate[k], ok = v.(string)
			if !ok {
				T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, fmt.Sprintf("malformed properties param")))
				return
			}
		} else {
			toDelete[k] = "delete"
		}
	}

	g, err := T.Backend.GameByName(ctx, params["name"].(string))

	if err == backend.ErrGameNotFound {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusNotFound, err.Error()))
		return
	} else if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return
	}

	err = T.Backend.UpdateProperties(ctx, g.ID, toUpdate)
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return
	}

	err = T.Backend.DeleteProperties(ctx, g.ID, toDelete)
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return
	}

	apiutils.ServeJSON(w, map[string]interface{}{"name": params["name"].(string), "properties": properties})
}
