package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performPatchRequest(body string) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("PATCH", baseURL, strings.NewReader(body))

	return makeTestRequestWithContext(router.updateProperties, r)
}

func TestUpdateProperties(t *testing.T) {
	Convey("should return error for missing params", t, func() {
		params := "{}"
		w := performPatchRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: name")

		params = fmt.Sprintf(`{"name":"%s"}`, backendtest.TestTerm)
		w = performPatchRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: properties")
	})

	Convey("error for bad properties", t, func() {
		params := fmt.Sprintf(`{"name":"%s","properties":{"a":"10","b":10}}`, backendtest.TestTerm)
		w := performPatchRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
	})

	Convey("shouldn't find game", t, func() {
		params := fmt.Sprintf(`{"name":"%s","properties":{"a":"10","b":"10"}}`, backendtest.TestBadTerm)
		w := performPatchRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})

	Convey("should work for correct params", t, func() {
		propstring := `{"a":"10","b":"10"}`
		params := fmt.Sprintf(`{"name":"%s","properties":%s}`, backendtest.TestTerm, propstring)
		props := map[string]interface{}{}

		json.Unmarshal([]byte(propstring), &props)

		w := performPatchRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)

		res := map[string]interface{}{}
		err := json.Unmarshal([]byte(w.Body.String()), &res)

		So(err, ShouldBeNil)
		So(res, ShouldResemble, map[string]interface{}{"name": backendtest.TestTerm, "properties": props})
	})
}
