package backend

import (
	"bytes"
	"encoding/csv"
	"errors"
	"fmt"
	"log"
	"strconv"
	"strings"
	"time"
	"unicode/utf8"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/config"
	"code.justin.tv/web/discovery/aws"
	"code.justin.tv/web/discovery/game"
	"code.justin.tv/web/discovery/kinesis"
	"code.justin.tv/web/discovery/sns"
	"code.justin.tv/web/discovery/streams"

	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

const (
	// DefaultOrdering is the ordering used to sort game results by the queried ids.
	DefaultOrdering = "default"
)

// Exported Error Variables
var (
	// ErrGameNotFound is raised when a game is not found in the DB for an id.
	ErrGameNotFound = fmt.Errorf("Game not found")
	// ErrLocalizationNotFound is raised when a localization is not found in the DB for a (game_id, locale) pair.
	ErrLocalizationNotFound = fmt.Errorf("Localization not found")
)

// Backender is the interface for the discovery backend.
type Backender interface {
	Health(ctx context.Context) error
	GameSuggest(ctx context.Context, term string, live bool) ([]game.Game, error)
	Top(ctx context.Context, limit, offset int, version int64, ids []int) ([]game.LiveGame, int, int64, error)
	GameIDMappingCSV(ctx context.Context) (string, error)
	AllAliasesCSV(ctx context.Context) (string, error)
	AliasesByGameID(ctx context.Context, id int) ([]string, error)
	GenresByGameID(ctx context.Context, id int) ([]string, error)
	GameByName(ctx context.Context, name string) (game.Game, error)
	GameByNameOrAlias(ctx context.Context, name string) (game.Game, error)
	GameByID(ctx context.Context, id int) (game.Game, error)
	GameByIDs(ctx context.Context, ids []int, ordering string) ([]game.Game, error)
	GameByAlias(ctx context.Context, alias string) (game.Game, error)
	GameByGBID(ctx context.Context, id int) (game.Game, error)
	AddAlias(ctx context.Context, id int, alias string) error
	AddGame(ctx context.Context, g game.Game) error
	UpdateGame(ctx context.Context, g game.Game) error
	DeleteGame(ctx context.Context, g game.Game) error
	GetPopularGameIDs(ctx context.Context) ([]int, error)
	UpdateGamePopularity(ctx context.Context, id, popularity int) error
	UpdateProperties(ctx context.Context, id int, properties map[string]string) error
	DeleteProperties(ctx context.Context, id int, properties map[string]string) error
	IsGameBlacklisted(ctx context.Context, id int) bool
	UpsertLocalization(ctx context.Context, gameID int, name, locale string) (game.Localizations, error)
	GetLocalization(ctx context.Context, gameID int, locale string) (game.Localizations, error)
	GetLocalizations(ctx context.Context, gameID int) (game.Localizations, error)
	GetBulkLocalizations(ctx context.Context, gameIDs []int, locale string) (map[int]game.Localizations, error)
	DeleteLocalization(ctx context.Context, gameID int, locale string) error
	DeleteLocalizations(ctx context.Context, gameID int) error
}

// Backend is an implementation of Backender.
type Backend struct {
	masterDB      *dbWrapper
	slaveDB       *dbWrapper
	streamsClient streams.Client
	stats         statsd.Statter
	gameIndexer   kinesis.GameIndexer
	gameCommunity sns.GameCommunity
}

var _ Backender = new(Backend)

func init() {
	config.Register(map[string]string{
		"searchindexer-stream-name": "searchindexer-staging",
		"gameCommunity-topic-arn":   "arn:aws:sns:us-west-2:465369119046:discovery_staging_game_addition_or_deletion",
	})
}

// NewBackend creates a Backender.
func NewBackend(streamsClient streams.Client, stats statsd.Statter) (*Backend, error) {
	masterOpts := MasterDBOptions()
	slaveOpts := SlaveDBOptions()

	master, err := db.Open(masterOpts...)
	if err != nil {
		log.Fatalf("error opening master db: %s", err.Error())
	}

	slave, err := db.Open(slaveOpts...)
	if err != nil {
		log.Fatalf("error opening slave db: %s", err.Error())
	}

	dbCb := func(evName, queryName string, d time.Duration) {
		bucket := fmt.Sprintf("%s.%s", evName, queryName)
		stats.Inc(bucket, 1, 0.1)
		stats.TimingDuration(bucket, d, 0.1)
	}
	runCb := func(evName string) {
		stats.Inc(evName, 1, 0.1)
	}
	master.SetCallbacks(dbCb, runCb)
	slave.SetCallbacks(dbCb, runCb)

	backend := Backend{
		masterDB:      &dbWrapper{master},
		slaveDB:       &dbWrapper{slave},
		streamsClient: streamsClient,
		stats:         stats,
		gameIndexer:   kinesis.NewGameIndexer(config.MustResolve("searchindexer-stream-name"), stats, aws.CreateAwsCredentials()),
		gameCommunity: sns.NewGameCommunity(config.MustResolve("gameCommunity-topic-arn"), stats, aws.CreateAwsCredentials()),
	}

	return &backend, nil
}

func normalizeName(name string) string {
	return strings.Replace(name, "’", "'", -1)
}

// Health records the state of database connections
func (B *Backend) Health(ctx context.Context) error {
	info := B.masterDB.Info()
	B.stats.Gauge("db.master.open_conns_cap", int64(info.OpenConnsCap), 0.1)
	B.stats.Gauge("db.master.max_open_conns", int64(info.MaxOpenConns), 0.1)
	B.stats.Gauge("db.master.min_available_conns", int64(info.MinAvailableConns), 0.1)

	info = B.slaveDB.Info()
	B.stats.Gauge("db.slave.open_conns_cap", int64(info.OpenConnsCap), 0.1)
	B.stats.Gauge("db.slave.max_open_conns", int64(info.MaxOpenConns), 0.1)
	B.stats.Gauge("db.slave.min_available_conns", int64(info.MinAvailableConns), 0.1)

	return nil
}

// Suggest returns a list of games whose names/aliases begin with a search term.
func (B *Backend) GameSuggest(ctx context.Context, term string, live bool) ([]game.Game, error) {
	term = strings.Replace(normalizeName(term), "%", "\\%", -1)
	term = strings.Replace(term, "_", "\\_", -1)

	if !utf8.ValidString(term) {
		return nil, errors.New(fmt.Sprintf("Non-UTF8 term: %s", term))
	}

	rows, err := B.slaveDB.Query(ctx, suggestGamesQuery(term))
	if err != nil {
		return nil, err
	}

	games := []game.Game{}

	for rows.Next() {
		g, err := ScanGame(rows)
		if err != nil {
			log.Printf("Failed to read a database row into a game: %s\n", err.Error())
		} else {
			games = append(games, g)
		}
	}
	rows.Close()

	var lives map[int]streams.JaxGame
	if live {
		lives, err = B.streamsClient.LiveGames(ctx)
		if err != nil {
			return nil, err
		}

		livegames := []game.Game{}
		for _, g := range games {
			if _, ok := lives[g.ID]; ok {
				livegames = append(livegames, g)
			}
		}
		return livegames, nil
	}

	return games, nil
}

// Top returns all live games, sorted in descending order of viewers
func (B *Backend) Top(ctx context.Context, limit, offset int, version int64, ids []int) ([]game.LiveGame, int, int64, error) {
	var err error
	var rows db.Rows
	var total int

	row := B.slaveDB.QueryRow(ctx, recentVersion(version))
	err = row.Scan(&version)
	if err != nil {
		return []game.LiveGame{}, 0, 0, err
	}

	if len(ids) == 0 {
		row := B.slaveDB.QueryRow(ctx, totalTopGames(version))
		err := row.Scan(&total)
		if err != nil {
			return []game.LiveGame{}, 0, 0, err
		}

		rows, err = B.slaveDB.Query(ctx, topGames(version, limit, offset))
	} else {
		total = len(ids)

		rows, err = B.slaveDB.Query(ctx, topGamesByID(version, ids, limit, offset))
	}
	if err != nil {
		return []game.LiveGame{}, 0, 0, err
	}

	finalGames := []game.LiveGame{}
	for rows.Next() {
		g, err := ScanLiveGame(rows)
		if err != nil {
			log.Printf("Could not scan game: %s\n", err.Error())
		}
		finalGames = append(finalGames, g)
	}
	rows.Close()

	return finalGames, total, version, nil
}

func (B *Backend) GameIDMappingCSV(ctx context.Context) (string, error) {
	rows, err := B.slaveDB.Query(ctx, gameIDMappingQuery())
	if err != nil {
		return "", err
	}

	buf := bytes.NewBufferString("")
	writer := csv.NewWriter(buf)
	writer.Write([]string{"id", "name", "giantbomb_id"})

	var id, name, gbID string
	for rows.Next() {
		err := rows.Scan(&id, &name, &gbID)
		if err != nil {
			log.Printf("Failed to scan mapping row: %s", err.Error())
		} else {
			writer.Write([]string{id, name, gbID})
		}
	}
	rows.Close()
	writer.Flush()

	return buf.String(), nil
}

func (B *Backend) AllAliasesCSV(ctx context.Context) (string, error) {
	rows, err := B.slaveDB.Query(ctx, allAliasesQuery())
	if err != nil {
		return "", err
	}

	buf := bytes.NewBufferString("")
	writer := csv.NewWriter(buf)
	writer.Write([]string{"game_id", "alias"})
	var id int
	var alias string
	for rows.Next() {
		err := rows.Scan(&id, &alias)
		if err != nil {
			log.Printf("Failed to scan alias row: %s", err.Error())
		} else {
			writer.Write([]string{strconv.Itoa(id), alias})
		}
	}
	rows.Close()
	writer.Flush()

	return buf.String(), nil
}

func (B *Backend) AliasesByGameID(ctx context.Context, id int) ([]string, error) {
	rows, err := B.slaveDB.Query(ctx, aliasesByGameIDQuery(id))
	if err != nil {
		return nil, err
	}

	aliases := []string{}
	var alias string
	for rows.Next() {
		err := rows.Scan(&alias)
		if err != nil {
			log.Printf("Failed to scan alias row: %s", err.Error())
		} else {
			aliases = append(aliases, alias)
		}
	}
	rows.Close()

	return aliases, nil
}

// Genres returns the genres of game specified by id.
func (B *Backend) GenresByGameID(ctx context.Context, id int) ([]string, error) {
	rows, err := B.slaveDB.Query(ctx, genresByGameIDQuery(id))
	if err != nil {
		return nil, err
	}

	genres := []string{}
	var genre string
	for rows.Next() {
		err := rows.Scan(&genre)
		if err != nil {
			log.Printf("Failed to scan genre row: %s", err.Error())
		} else {
			genres = append(genres, genre)
		}
	}
	rows.Close()

	return genres, nil
}

// gameByQuery is a helper function that queries the database for a single game using given query.
func (B *Backend) gameByQuery(ctx context.Context, query dbQuery) (game.Game, error) {
	g := game.Game{}

	rows, err := B.slaveDB.Query(ctx, query)
	if err != nil {
		return g, err
	}

	if !rows.Next() {
		return g, ErrGameNotFound
	}

	g, err = ScanGame(rows)
	rows.Close()
	if err != nil {
		return g, err
	}

	return g, nil
}

// GameByName retrieves a game from the database using the game name.
func (B *Backend) GameByName(ctx context.Context, name string) (game.Game, error) {
	return B.gameByQuery(ctx, searchName(normalizeName(name)))
}

// GameByNameOrAlias retrieves a game from the database using the game name or alias.
func (B *Backend) GameByNameOrAlias(ctx context.Context, name string) (game.Game, error) {
	return B.gameByQuery(ctx, searchNameOrAlias(normalizeName(name)))
}

// GameByID retrieves a game from the database using the id.
func (B *Backend) GameByID(ctx context.Context, id int) (game.Game, error) {
	return B.gameByQuery(ctx, searchID(id))
}

// GameByIDs retrieves several games from the database using their ids.
// Results can be ordered by the game name or by the order the ids were given.
func (B *Backend) GameByIDs(ctx context.Context, ids []int, ordering string) ([]game.Game, error) {
	games := []game.Game{}

	if len(ids) == 0 {
		return games, nil
	}

	rows, err := B.slaveDB.Query(ctx, searchIDs(ids, ordering))
	if err != nil {
		return games, err
	}

	gameLookup := map[int]int{}

	for rows.Next() {
		g, err := ScanGame(rows)
		if err == nil {
			games = append(games, g)
		}
		gameLookup[g.ID] = len(games) - 1
	}
	rows.Close()

	if ordering == DefaultOrdering {
		orderedGames := []game.Game{}
		for _, id := range ids {
			gameIndex, gameExists := gameLookup[id]
			if gameExists {
				g := games[gameIndex]
				if g.ID == id {
					orderedGames = append(orderedGames, g)
				}
			}
		}
		games = orderedGames
	}

	return games, nil
}

// GameByAlias retrieves a game from the database using an alias.
func (B *Backend) GameByAlias(ctx context.Context, alias string) (game.Game, error) {
	return B.gameByQuery(ctx, searchAlias(normalizeName(alias)))
}

// GameByGBID retrieves a game from the database using the giantbomb id.
func (B *Backend) GameByGBID(ctx context.Context, id int) (game.Game, error) {
	return B.gameByQuery(ctx, searchGBID(id))
}

// AddAlias assigns an alias to a game by id.
func (B *Backend) AddAlias(ctx context.Context, id int, alias string) error {
	_, err := B.masterDB.Exec(ctx, insertAliasQuery(id, normalizeName(alias)))
	B.indexGame(ctx, id)
	return err
}

// AddGame inserts a new game into the database.
func (B *Backend) AddGame(ctx context.Context, g game.Game) error {
	if g.GiantbombID == 0 {
		_, err := B.masterDB.Exec(ctx, insertGameNoGBID(
			g.Popularity,
			normalizeName(g.Name),
			mapToHstore(g.Images),
			mapToHstore(g.Properties),
		))

		game, gErr := B.GameByName(ctx, g.Name)
		if gErr == nil {
			B.indexGame(ctx, game.ID)
			B.gameCommunity.UpdateGame(game.ID, g.Name, "add")
		}

		return err
	}

	_, err := B.masterDB.Exec(ctx, insertGameQuery(
		g.GiantbombID,
		g.Popularity,
		normalizeName(g.Name),
		mapToHstore(g.Images),
		mapToHstore(g.Properties),
	))

	game, gErr := B.GameByName(ctx, g.Name)
	if gErr == nil {
		B.indexGame(ctx, game.ID)
		B.gameCommunity.UpdateGame(game.ID, g.Name, "add")
	}

	return err
}

// UpdateGame replaces a game in the database.
func (B *Backend) UpdateGame(ctx context.Context, g game.Game) error {
	_, err := B.masterDB.Exec(ctx, updateGameQuery(
		g.ID,
		g.GiantbombID,
		g.Popularity,
		normalizeName(g.Name),
		mapToHstore(g.Images),
		mapToHstore(g.Properties),
	))
	B.indexGame(ctx, g.ID)
	return err
}

// DeleteGame deletes a game from the database.
func (B *Backend) DeleteGame(ctx context.Context, g game.Game) error {
	_, err := B.masterDB.Exec(ctx, deleteGameQuery(g.ID))
	if err != nil {
		return err
	}

	B.deleteGameIndex(ctx, g.ID)
	B.gameCommunity.UpdateGame(g.ID, "", "delete")
	if g.GiantbombID > 0 {
		_, err = B.masterDB.Exec(ctx, blacklistGameQuery(g.GiantbombID))
	}
	return err
}

func (B *Backend) GetPopularGameIDs(ctx context.Context) ([]int, error) {
	rows, err := B.slaveDB.Query(ctx, popularGamesQuery())
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var id int
	ids := []int{}
	for rows.Next() {
		if err := rows.Scan(&id); err == nil {
			ids = append(ids, id)
		}
	}

	return ids, nil
}

// UpdateGamePopularity updates a game's popularity value in the database.
func (B *Backend) UpdateGamePopularity(ctx context.Context, id, popularity int) error {
	_, err := B.masterDB.Exec(ctx, updatePopularityQuery(
		id,
		popularity,
	))
	B.indexGame(ctx, id)
	return err
}

// UpdateProperties sets the properties for a game id.
func (B *Backend) UpdateProperties(ctx context.Context, id int, properties map[string]string) error {
	_, err := B.masterDB.Exec(ctx, updatePropertiesQuery(id, mapToHstore(properties)))
	return err
}

// DeleteProperties deletes the properties for a game id.
func (B *Backend) DeleteProperties(ctx context.Context, id int, properties map[string]string) error {
	if len(properties) == 0 {
		return nil
	}
	_, err := B.masterDB.Exec(ctx, deletePropertiesQuery(id, properties))
	return err
}

// IsGameBlacklisted returns whether a giantbomb id is blacklisted from being updated from giantbomb.
func (B *Backend) IsGameBlacklisted(ctx context.Context, giantbombID int) bool {
	r := B.slaveDB.QueryRow(ctx, isBlacklistQuery(giantbombID))
	var exists bool
	err := r.Scan(&exists)
	if err != nil {
		return true
	}
	return exists
}

// UpsertLocalization upserts a localization into the database.
func (B *Backend) UpsertLocalization(ctx context.Context, gameID int, name, locale string) (game.Localizations, error) {
	localizations := game.Localizations{LocalizedNames: make(map[string]string)}

	rows, err := B.masterDB.Query(ctx, upsertLocalizationQuery(
		gameID,
		normalizeName(name),
		locale,
	))
	if err != nil {
		return localizations, err
	}

	B.indexGame(ctx, gameID)

	if !rows.Next() {
		return localizations, ErrLocalizationNotFound
	}

	l, err := ScanLocalization(rows)
	rows.Close()
	if err != nil {
		return localizations, err
	}

	localizations.GameID = l.GameID
	localizations.LocalizedNames[l.Locale] = l.Name

	return localizations, nil
}

// GetLocalization retrieves a localization from the database for a given gameID and locale.
func (B *Backend) GetLocalization(ctx context.Context, gameID int, locale string) (game.Localizations, error) {
	localizations := game.Localizations{LocalizedNames: make(map[string]string)}

	rows, err := B.slaveDB.Query(ctx, getLocalizationQuery(
		gameID,
		locale,
	))
	if err != nil {
		return localizations, err
	}

	if !rows.Next() {
		return localizations, ErrLocalizationNotFound
	}

	l, err := ScanLocalization(rows)
	rows.Close()
	if err != nil {
		return localizations, err
	}

	localizations.GameID = l.GameID
	localizations.LocalizedNames[l.Locale] = l.Name

	return localizations, nil
}

// GetLocalizations retrieves localizations from the database for a given gameID.
func (B *Backend) GetLocalizations(ctx context.Context, gameID int) (game.Localizations, error) {
	localizations := game.Localizations{
		GameID:         gameID,
		LocalizedNames: make(map[string]string),
	}

	rows, err := B.slaveDB.Query(ctx, getLocalizationsQuery(
		gameID,
	))
	if err != nil {
		return localizations, err
	}

	l := game.LocalizationData{}

	for rows.Next() {
		l, err = ScanLocalization(rows)
		if err == nil {
			localizations.LocalizedNames[l.Locale] = l.Name
		}
	}

	rows.Close()

	return localizations, nil
}

func (B *Backend) GetBulkLocalizations(ctx context.Context, gameIDs []int, locale string) (map[int]game.Localizations, error) {
	localizations := map[int]game.Localizations{}

	if len(gameIDs) == 0 {
		return localizations, nil
	}

	rows, err := B.slaveDB.Query(ctx, getBulkLocalizationsQuery(
		gameIDs,
		locale,
	))
	if err != nil {
		return nil, err
	}

	for rows.Next() {
		l, err := ScanLocalization(rows)
		if err == nil {
			localization, ok := localizations[l.GameID]
			if !ok {
				localization = game.Localizations{
					GameID:         l.GameID,
					LocalizedNames: map[string]string{},
				}
			}

			localization.LocalizedNames[locale] = l.Name
			localizations[l.GameID] = localization
		}
	}
	rows.Close()

	return localizations, nil
}

// DeleteLocalization deletes a localization from the database for a given gameID and locale.
func (B *Backend) DeleteLocalization(ctx context.Context, gameID int, locale string) error {
	_, err := B.masterDB.Exec(ctx, deleteLocalizationQuery(
		gameID,
		locale,
	))
	if err != nil {
		return err
	}

	B.indexGame(ctx, gameID)
	return err
}

// DeleteLocalization deletes localizations from the database for a given gameID.
func (B *Backend) DeleteLocalizations(ctx context.Context, gameID int) error {
	_, err := B.masterDB.Exec(ctx, deleteLocalizationsQuery(
		gameID,
	))
	if err != nil {
		return err
	}

	B.indexGame(ctx, gameID)
	return err
}

func (B *Backend) indexGame(ctx context.Context, gameID int) error {
	game, err := B.GameByID(ctx, gameID)
	if err != nil {
		return err
	}
	aliases, err := B.AliasesByGameID(ctx, gameID)
	if err != nil {
		return err
	}

	localizations, err := B.GetLocalizations(ctx, gameID)
	if err != nil {
		if err != ErrLocalizationNotFound {
			return err
		}
	}
	return B.gameIndexer.AddGame(game.ID, game.Popularity, game.Name, aliases, localizations.LocalizedNames)
}

func (B *Backend) deleteGameIndex(ctx context.Context, gameID int) error {
	return B.gameIndexer.DeleteGame(gameID)
}
