package backend

import (
	"database/sql"
	"fmt"
	"strconv"
	"testing"

	"code.justin.tv/chat/db/dbtest"
	"code.justin.tv/web/discovery/game"
	"code.justin.tv/web/discovery/kinesis/kinesistest"
	"code.justin.tv/web/discovery/sns/snstest"
	"code.justin.tv/web/discovery/streams/streamstest"
	"github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

const TestBadGameID = 1345

var aliasColumns = []string{"alias"}
var gameColumns = []string{"id", "giantbomb_id", "name", "images", "popularity", "properites"}
var livegameColumns = []string{"viewers", "channels", "id", "giantbomb_id", "name", "images", "popularity", "properites"}
var genreColumns = []string{"name"}
var localizationColumns = []string{"game_id", "name", "locale"}

var testLocalizationData = game.LocalizationData{
	GameID: 123,
	Name:   "Liga de Leyendas",
	Locale: "es",
}

var testLocalization = game.Localizations{
	GameID:         123,
	LocalizedNames: map[string]string{"es": "Liga de Leyendas"},
}

func NewMockDB() (*sql.DB, sqlmock.Sqlmock) {
	mockDB, mock, err := sqlmock.New()
	if err != nil {
		panic("An error was not expected when opening a stub connection: " + err.Error())
	}

	return mockDB, mock
}

func TestNormalize(t *testing.T) {
	badName := "Test’s Game"
	goodName := "Test's Game"

	Convey("should keep names without fancy quotes unchanged", t, func() {
		So(normalizeName(goodName), ShouldEqual, goodName)
	})

	Convey("should convert fancy quotes into normal quotes", t, func() {
		So(normalizeName(badName), ShouldEqual, goodName)
	})
}

func TestGameSuggest(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockJax := streamstest.NewStreamsClient()
	b := &Backend{slaveDB: &dbWrapper{mockDB}, streamsClient: mockJax}

	gameName := "starlord"
	gameID := streamstest.TestIDs[0]
	mock.ExpectQuery("SELECT (.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString("1,2," + gameName + ",small=>link_to_pic,3,\n1,2," + gameName + ",small=>link_to_pic,3,"))

	games, err := b.GameSuggest(context.Background(), "star", false)

	Convey("should not be error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return two games", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 2)
	})

	Convey("should return the right games", t, func() {
		So(games[0].Name, ShouldEqual, gameName)
		So(games[1].Name, ShouldEqual, gameName)
	})

	mock.ExpectQuery("SELECT (.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString(strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,\n1,2," + gameName + ",small=>link_to_pic,3,"))

	games, err = b.GameSuggest(context.Background(), "star", true)
	Convey("should find the right live game", t, func() {
		So(err, ShouldBeNil)
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 1)
		So(games[0].Name, ShouldEqual, gameName)
	})

	games, err = b.GameSuggest(context.Background(), string([]byte{0xff}), false)
	Convey("should return an error", t, func() {
		So(err, ShouldNotBeNil)
		So(games, ShouldBeNil)
	})
}

func TestTop(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockJax := streamstest.NewStreamsClient()
	b := &Backend{slaveDB: &dbWrapper{mockDB}, streamsClient: mockJax}

	gameName := "smite"
	gameID := streamstest.TestIDs[1]
	version := 14
	total := 1

	mock.ExpectQuery("SELECT COALESCE(.+)").
		WillReturnRows(sqlmock.NewRows([]string{"coalesce"}).
			FromCSVString(strconv.Itoa(version)))

	mock.ExpectQuery("SELECT COUNT(.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows([]string{"count"}).
			FromCSVString(strconv.Itoa(total)))

	mock.ExpectQuery("SELECT (.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows(livegameColumns).
			FromCSVString("1,2," + strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,"))

	games, totalGames, v, err := b.Top(context.Background(), 10, 0, 0, []int{})

	Convey("should not be error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return one game", t, func() {
		So(games, ShouldNotBeNil)
		So(totalGames, ShouldEqual, total)
		So(len(games), ShouldEqual, 1)
		So(v, ShouldEqual, version)
		So(games[0].Game.Name, ShouldEqual, gameName)
	})
}

func TestGameIDMappingCSV(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows([]string{"id", "name", "giantbomb_id"}).FromCSVString(
			`1,"name",2
			2,"name2,version2",3`))

	games, err := b.GameIDMappingCSV(context.Background())

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right games", t, func() {
		So(games, ShouldEqual,
			`id,name,giantbomb_id
1,name,2
2,"name2,version2",3
`)
	})
}

func TestAllAliasesCSV(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	mock.ExpectQuery("SELECT (.+) FROM game_aliases").
		WillReturnRows(sqlmock.NewRows([]string{"game_id", "alias"}).FromCSVString(
			`1,"name"
			2,"name2,version2"`))

	aliases, err := b.AllAliasesCSV(context.Background())

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right aliases", t, func() {
		So(aliases, ShouldEqual,
			`game_id,alias
1,name
2,"name2,version2"
`)
	})
}

func TestAliasesByGameID(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameID := 1234
	gameAlias := "test"
	mock.ExpectQuery("SELECT (.+) FROM game_aliases (.+)").
		WillReturnRows(sqlmock.NewRows(aliasColumns).FromCSVString(gameAlias))

	genres, err := b.AliasesByGameID(context.Background(), gameID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right genres", t, func() {
		So(genres, ShouldContain, gameAlias)
	})
}

func TestGenresByGameID(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameID := 1234
	gameGenre := "platformer"
	mock.ExpectQuery("SELECT (.+) FROM genres (.+)").
		WillReturnRows(sqlmock.NewRows(genreColumns).FromCSVString(gameGenre))

	genres, err := b.GenresByGameID(context.Background(), gameID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right genres", t, func() {
		So(genres, ShouldContain, gameGenre)
	})
}

func TestGameByQuery(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameName := "braid"
	gameID := 1234

	query := "SELECT foo FROM bar"
	args := []string{"a", "b", "c"}
	mock.ExpectQuery(query).WithArgs(args[0], args[1], args[2]).
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,"))

	g, err := b.gameByQuery(context.Background(), dbQuery{
		query: query,
		args:  []interface{}{args[0], args[1], args[2]}})

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a game", t, func() {
		So(g, ShouldNotBeNil)
	})

	Convey("should return the right game", t, func() {
		So(g.Name, ShouldEqual, gameName)
	})
}

func TestGameByIDs(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameNames := []string{"braid", "abraid"}
	gameIDs := []int{1234, 5678}
	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString(strconv.Itoa(gameIDs[0]) + ",1," + gameNames[0] + ",small=>link_to_pic,3,\n" +
				strconv.Itoa(gameIDs[1]) + ",2," + gameNames[1] + ",small=>link_to_pic,4,"))

	games, err := b.GameByIDs(context.Background(), gameIDs, DefaultOrdering)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return games", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 2)
	})

	Convey("should return the right games", t, func() {
		So(games[0].Name, ShouldEqual, gameNames[0])
		So(games[1].Name, ShouldEqual, gameNames[1])
	})

	mock.ExpectQuery("SELECT (.+) FROM games (.+) ORDER BY name").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString(strconv.Itoa(gameIDs[1]) + ",1," + gameNames[1] + ",small=>link_to_pic,3,\n" +
				strconv.Itoa(gameIDs[0]) + ",2," + gameNames[0] + ",small=>link_to_pic,4,"))

	games, err = b.GameByIDs(context.Background(), gameIDs, "name")

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return games", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 2)
	})

	Convey("should return the right games", t, func() {
		So(games[0].Name, ShouldEqual, gameNames[1])
		So(games[1].Name, ShouldEqual, gameNames[0])
	})

	gameIDs = []int{1234, 3456, 5678}
	gameNames = []string{"braid", "fake", "abraid"}

	mock.ExpectQuery("SELECT (.+) FROM games (.+) ORDER BY name").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString(strconv.Itoa(gameIDs[2]) + ",1," + gameNames[2] + ",small=>link_to_pic,3,\n" +
				strconv.Itoa(gameIDs[0]) + ",2," + gameNames[0] + ",small=>link_to_pic,4,"))

	games, err = b.GameByIDs(context.Background(), gameIDs, "name")

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return games", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 2)
	})

	Convey("should return the existing games", t, func() {
		So(games[0].Name, ShouldEqual, gameNames[2])
		So(games[1].Name, ShouldEqual, gameNames[0])
	})

	gameIDs = []int{1234}

	mock.ExpectQuery("SELECT (.+) FROM games (.+) ORDER BY name").
		WillReturnRows(sqlmock.NewRows(gameColumns).
			FromCSVString(""))

	games, err = b.GameByIDs(context.Background(), gameIDs, "name")

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return empty list when no existing game ids provided", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 0)
	})

	games, err = b.GameByIDs(context.Background(), []int{}, "name")

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return empty list when no ids provided", t, func() {
		So(games, ShouldNotBeNil)
		So(len(games), ShouldEqual, 0)
	})
}

func TestGameByID(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameName := "braid"
	gameID := 1234
	mock.ExpectQuery("SELECT (.+) FROM games (.+)").WithArgs(gameID).
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,"))

	games, err := b.GameByID(context.Background(), gameID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a game", t, func() {
		So(games, ShouldNotBeNil)
	})

	Convey("should return the right game", t, func() {
		So(games.Name, ShouldEqual, gameName)
	})

	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(""))

	games, err = b.GameByID(context.Background(), TestBadGameID)

	Convey("should not find a bad game id", t, func() {
		So(err, ShouldNotBeNil)
		So(err.Error(), ShouldContainSubstring, "not found")
	})
}

func TestGameByName(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameName := "braid"
	gameID := 1234
	mock.ExpectQuery("SELECT (.+) FROM games (.+)").WithArgs(gameName).
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,"))

	games, err := b.GameByName(context.Background(), gameName)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a game", t, func() {
		So(games, ShouldNotBeNil)
	})

	Convey("should return the right game", t, func() {
		So(games.ID, ShouldEqual, gameID)
	})

	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(""))

	games, err = b.GameByName(context.Background(), "asdlkasjdkl")

	Convey("should not find a bad game name", t, func() {
		So(err, ShouldNotBeNil)
		So(err.Error(), ShouldContainSubstring, "not found")
	})
}

func TestGameByAlias(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameName := "braid"
	gameAlias := "braid"
	gameID := 1234
	mock.ExpectQuery("SELECT (.+) FROM games (.+)").WithArgs(gameAlias).
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(strconv.Itoa(gameID) + ",2," + gameName + ",small=>link_to_pic,3,"))

	games, err := b.GameByAlias(context.Background(), gameAlias)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a game", t, func() {
		So(games, ShouldNotBeNil)
	})

	Convey("should return the right game", t, func() {
		So(games.ID, ShouldEqual, gameID)
	})

	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(""))

	games, err = b.GameByAlias(context.Background(), "asdlkasjdkl")

	Convey("should not find a bad game alias", t, func() {
		So(err, ShouldNotBeNil)
		So(err.Error(), ShouldContainSubstring, "not found")
	})
}

func TestGameByGBID(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	gameName := "braid"
	gameGBID := 1234
	mock.ExpectQuery("SELECT (.+) FROM games (.+)").WithArgs(gameGBID).
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString("1," + strconv.Itoa(gameGBID) + "," + gameName + ",small=>link_to_pic,3,"))

	games, err := b.GameByGBID(context.Background(), gameGBID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a game", t, func() {
		So(games, ShouldNotBeNil)
	})

	Convey("should return the right game", t, func() {
		So(games.GiantbombID, ShouldEqual, gameGBID)
	})

	mock.ExpectQuery("SELECT (.+) FROM games (.+)").
		WillReturnRows(sqlmock.NewRows(gameColumns).FromCSVString(""))

	games, err = b.GameByGBID(context.Background(), 0)

	Convey("should not find a bad game gbid", t, func() {
		So(err, ShouldNotBeNil)
		So(err.Error(), ShouldContainSubstring, "not found")
	})
}

func TestAddAlias(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	gameAlias := "braid"
	gameID := 1234
	mock.ExpectExec("INSERT INTO game_aliases (.+)").WithArgs(gameAlias, gameID).WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.AddAlias(context.Background(), gameID, gameAlias)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestAddGame(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	mockGameCommunity := snstest.NewGameCommunity()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer, gameCommunity: mockGameCommunity}

	g := game.Game{
		Name:        "braid",
		GiantbombID: 1234,
		Images:      make(map[string]string),
		Properties:  make(map[string]string),
	}

	hs := []string{}

	mock.ExpectExec("INSERT INTO games (.+)").
		WithArgs(g.GiantbombID, g.Name, 0, hs, hs).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.AddGame(context.Background(), g)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestAddGame2(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	g := game.Game{
		Name:       "braid",
		Images:     make(map[string]string),
		Properties: make(map[string]string),
	}

	hs := []string{}

	mock.ExpectExec("INSERT INTO games (.+)").
		WithArgs(g.Name, 0, hs, hs).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.AddGame(context.Background(), g)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestUpdateGame(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	g := game.Game{
		ID:          1,
		Name:        "braid",
		GiantbombID: 1234,
		Images:      make(map[string]string),
		Popularity:  0,
		Properties:  make(map[string]string),
	}

	hs := []string{}

	mock.ExpectExec("UPDATE games (.+)").
		WithArgs(g.ID, g.GiantbombID, g.Popularity, g.Name, hs, hs).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.UpdateGame(context.Background(), g)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestDeleteGame(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	mockGameCommunity := snstest.NewGameCommunity()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer, gameCommunity: mockGameCommunity}

	g := game.Game{
		ID:          1,
		Name:        "braid",
		GiantbombID: 1234,
		Images:      make(map[string]string),
		Popularity:  0,
		Properties:  make(map[string]string),
	}

	mock.ExpectExec("DELETE FROM games (.+)").
		WithArgs(g.ID).
		WillReturnResult(sqlmock.NewResult(0, 1))

	mock.ExpectExec("INSERT INTO giantbomb_blacklist (.+)").
		WithArgs(g.GiantbombID).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.DeleteGame(context.Background(), g)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestGetPopularGameIDs(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	mock.ExpectQuery("SELECT id FROM games WHERE popularity > 0").
		WillReturnRows(sqlmock.NewRows([]string{"id"}).FromCSVString("1"))

	_, err := b.GetPopularGameIDs(context.Background())

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestUpdateGamePopularity(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	g := game.Game{
		ID:          1,
		Name:        "braid",
		GiantbombID: 1234,
		Images:      make(map[string]string),
		Popularity:  0,
		Properties:  make(map[string]string),
	}

	mock.ExpectExec("UPDATE games (.+)").
		WithArgs(g.ID, g.Popularity).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.UpdateGamePopularity(context.Background(), g.ID, g.Popularity)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestUpdateProperties(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}}

	g := game.Game{
		Name:       "braid",
		Properties: make(map[string]string),
	}

	hs := []string{}

	mock.ExpectExec("UPDATE games (.+)").
		WithArgs(g.ID, hs).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.UpdateProperties(context.Background(), g.ID, g.Properties)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestDeleteProperties(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}}

	g := game.Game{
		Name:       "braid",
		Properties: map[string]string{"hello": "world"},
	}

	mock.ExpectExec("UPDATE games (.+)").
		WithArgs(g.ID).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.DeleteProperties(context.Background(), g.ID, g.Properties)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestUpsertLocalization(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	mock.ExpectQuery("INSERT INTO localizations (.+)").
		WithArgs(testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale).
		WillReturnRows(sqlmock.NewRows(localizationColumns).
			FromCSVString(fmt.Sprintf("%v,%v,%v", testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale)))

	localization, err := b.UpsertLocalization(context.Background(), testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a localization", t, func() {
		So(localization, ShouldNotBeNil)
	})

	Convey("should return the right localization", t, func() {
		So(localization.GameID, ShouldEqual, testLocalization.GameID)
		So(localization.LocalizedNames, ShouldResemble, testLocalization.LocalizedNames)
	})
}

func TestGetLocalization(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WithArgs(testLocalizationData.GameID, testLocalizationData.Locale).
		WillReturnRows(sqlmock.NewRows(localizationColumns).
			FromCSVString(fmt.Sprintf("%v,%v,%v", testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale)))

	localization, err := b.GetLocalization(context.Background(), testLocalizationData.GameID, testLocalizationData.Locale)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return a localization", t, func() {
		So(localization, ShouldNotBeNil)
	})

	Convey("should return the right localization", t, func() {
		So(localization.GameID, ShouldEqual, testLocalization.GameID)
		So(localization.LocalizedNames, ShouldResemble, testLocalization.LocalizedNames)
	})

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WillReturnRows(sqlmock.NewRows(localizationColumns).FromCSVString(""))

	localization, err = b.GetLocalization(context.Background(), TestBadGameID, "abc")

	Convey("should not find a localization not present", t, func() {
		So(err, ShouldNotBeNil)
		So(err.Error(), ShouldContainSubstring, "Localization not found")
	})
}

func TestGetLocalizations(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WithArgs(testLocalizationData.GameID).
		WillReturnRows(sqlmock.NewRows(localizationColumns).
			FromCSVString(fmt.Sprintf("%v,%v,%v", testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale)))

	localization, err := b.GetLocalizations(context.Background(), testLocalizationData.GameID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right localization", t, func() {
		So(localization.GameID, ShouldEqual, testLocalization.GameID)
		So(localization.LocalizedNames, ShouldResemble, testLocalization.LocalizedNames)
	})

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WillReturnRows(sqlmock.NewRows(localizationColumns).FromCSVString(""))

	localization, err = b.GetLocalizations(context.Background(), TestBadGameID)

	Convey("should return empty Localizations", t, func() {
		So(err, ShouldBeNil)
		So(localization.GameID, ShouldEqual, TestBadGameID)
		So(localization.LocalizedNames, ShouldBeEmpty)
	})
}

func TestGetBulkLocalizations(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	b := &Backend{slaveDB: &dbWrapper{mockDB}}

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WithArgs(testLocalizationData.Locale).
		WillReturnRows(sqlmock.NewRows(localizationColumns).
			FromCSVString(fmt.Sprintf("%v,%v,%v", testLocalizationData.GameID, testLocalizationData.Name, testLocalizationData.Locale)))

	localizations, err := b.GetBulkLocalizations(context.Background(), []int{testLocalizationData.GameID}, testLocalizationData.Locale)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return the right localization", t, func() {
		localization, ok := localizations[testLocalization.GameID]

		So(ok, ShouldBeTrue)
		So(localization.GameID, ShouldEqual, testLocalization.GameID)
		So(localization.LocalizedNames, ShouldResemble, testLocalization.LocalizedNames)
	})

	localizations, err = b.GetBulkLocalizations(context.Background(), []int{}, testLocalizationData.Locale)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should return empty list for no gameIDs passed", t, func() {
		So(len(localizations), ShouldEqual, 0)
	})

	mock.ExpectQuery("SELECT (.+) FROM localizations (.+)").
		WillReturnRows(sqlmock.NewRows(localizationColumns).FromCSVString(""))

	localizations, err = b.GetBulkLocalizations(context.Background(), []int{TestBadGameID}, "")

	Convey("should not find a localization not present", t, func() {
		_, ok := localizations[testLocalization.GameID]

		So(ok, ShouldBeFalse)
	})
}

func TestDeleteLocalization(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	mock.ExpectExec("DELETE FROM localizations (.+)").
		WithArgs(testLocalizationData.GameID, testLocalizationData.Locale).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.DeleteLocalization(context.Background(), testLocalizationData.GameID, testLocalizationData.Locale)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}

func TestDeleteLocalizations(t *testing.T) {
	mockSQL, mock := NewMockDB()
	mockDB := dbtest.NewMockDB(mockSQL)
	mockGameIndexer := kinesistest.NewGameIndexer()
	b := &Backend{masterDB: &dbWrapper{mockDB}, slaveDB: &dbWrapper{mockDB}, gameIndexer: mockGameIndexer}

	mock.ExpectExec("DELETE FROM localizations (.+)").
		WithArgs(testLocalization.GameID).
		WillReturnResult(sqlmock.NewResult(0, 1))

	err := b.DeleteLocalizations(context.Background(), testLocalization.GameID)

	Convey("should not have an error", t, func() {
		So(err, ShouldBeNil)
	})
}
