package backendtest

import (
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/game"
	"golang.org/x/net/context"
)

type Backend struct {
}

var _ backend.Backender = new(Backend)

const (
	TestBadTerm   = "123jljklj1kl23"
	TestTerm      = "braid"
	TestTerm2     = "overwatch"
	TestTerm3     = "league"
	TestID        = 123
	TestID2       = 456
	TestID3       = 789
	TestImageURL  = "imgur.com/abcde"
	TestImageName = "small"

	TestLimit    = 5
	TestOffset   = 3
	TestViewers  = 10
	TestChannels = 100
	TestNumGames = 30
	// Must match array below
	TestIDsString = "1,2,3,4,5,6,7,8,9,10"
	TestBadIDs    = `1,2,3,4\`

	TestAlias  = "imconfused"
	TestAlias2 = TestTerm3

	TestGBID            = 2600
	TestGBID2           = 3600
	TestGBID3           = 4600
	TestBlacklistedGBID = 934785

	TestLocalizedName = "Liga de Leyendas"
	TestLocale        = "es-es"
	TestEmptyLocale   = ""
)

var TestGameIDMapping = "id,name,giantbomb_id\n1,Test,1"
var TestAllAliases = "game_id,alias\n1,Test"

var TestAliases = []string{TestAlias}

var TestIDs = []int{1, 2, 3, 4, 5, 6, 7, 8, 9, 10}

var TestGenres = []string{"platformer", "puzzle"}

var TestLocalizedNames = map[string]string{TestLocale: TestLocalizedName}

func TestGame(id int) game.Game {
	name := TestTerm
	gbid := TestGBID

	switch id {
	case TestID:
		name = TestTerm
		gbid = TestGBID
	case TestID2:
		name = TestTerm2
		gbid = TestGBID2
	case TestID3:
		name = TestTerm3
		gbid = TestGBID3
	}

	return game.Game{
		Name:        name,
		ID:          id,
		GiantbombID: gbid,
		Images:      map[string]string{TestImageName: TestImageURL},
		Popularity:  TestID,
		Properties:  map[string]string{TestImageName: TestImageURL},
		BoxArt:      map[string]string{TestImageName: TestImageURL},
		LogoArt:     map[string]string{TestImageName: TestImageURL},
	}
}

func TestLocalization(gameID int, localizedName, locale string) game.Localizations {
	return game.Localizations{
		GameID:         gameID,
		LocalizedNames: map[string]string{locale: localizedName},
	}
}

func (T *Backend) Health(ctx context.Context) error {
	return nil
}

func (T *Backend) GameSuggest(ctx context.Context, term string, live bool) ([]game.Game, error) {
	switch term {
	case TestTerm:
		return []game.Game{TestGame(TestID)}, nil
	case TestTerm2:
		return []game.Game{TestGame(TestID2)}, nil
	case TestTerm3:
		return []game.Game{TestGame(TestID3)}, nil
	default:
		return []game.Game{}, nil
	}
}

func (T *Backend) Top(ctx context.Context, limit, offset int, version int64, ids []int) ([]game.LiveGame, int, int64, error) {
	liveGames := []game.LiveGame{}
	for i := 0; i < TestNumGames; i++ {
		if len(ids) == 0 {
			liveGames = append(liveGames,
				game.LiveGame{
					Channels: TestChannels,
					Viewers:  TestViewers,
					Game:     TestGame(i),
				})
		} else {
			// Check that the game is in the id list
			for _, j := range ids {
				if i == j {
					liveGames = append(liveGames,
						game.LiveGame{
							Channels: TestChannels,
							Viewers:  TestViewers,
							Game:     TestGame(i),
						})
				}
			}
		}
	}

	if offset > len(liveGames) {
		return []game.LiveGame{}, len(liveGames), 0, nil
	} else if offset+limit >= len(liveGames) {
		return liveGames[offset:], len(liveGames), 0, nil
	} else {
		return liveGames[offset : offset+limit], len(liveGames), 0, nil
	}
}

func (T *Backend) GameIDMappingCSV(ctx context.Context) (string, error) {
	return TestGameIDMapping, nil
}

func (T *Backend) AllAliasesCSV(ctx context.Context) (string, error) {
	return TestAllAliases, nil
}

func (T *Backend) AliasesByGameID(ctx context.Context, id int) ([]string, error) {
	return TestAliases, nil
}

func (T *Backend) GenresByGameID(ctx context.Context, id int) ([]string, error) {
	return TestGenres, nil
}

func (T *Backend) GameByName(ctx context.Context, name string) (game.Game, error) {
	switch name {
	case TestTerm:
		return TestGame(TestID), nil
	case TestTerm2:
		return TestGame(TestID2), nil
	case TestTerm3:
		return TestGame(TestID3), nil
	default:
		return game.Game{}, backend.ErrGameNotFound
	}
}

func (T *Backend) GameByNameOrAlias(ctx context.Context, name string) (game.Game, error) {
	if name == TestAlias || name == TestTerm {
		return TestGame(TestID), nil
	}

	// match TestAlias2 before TestTerm3 to force error case of GameByNameOrAlias
	// not prioritizing exact name matches over alias matches
	if name == TestAlias2 || name == TestTerm2 {
		return TestGame(TestID2), nil
	}

	if name == TestTerm3 {
		return TestGame(TestID3), nil
	}

	return game.Game{}, backend.ErrGameNotFound
}

func (T *Backend) GameByID(ctx context.Context, id int) (game.Game, error) {
	switch id {
	case TestID:
		return TestGame(TestID), nil
	case TestID2:
		return TestGame(TestID2), nil
	case TestID3:
		return TestGame(TestID3), nil
	default:
		return game.Game{}, backend.ErrGameNotFound
	}
}

// in returns whether an int is in a list of int.
func in(val int, list []int) bool {
	for _, v := range list {
		if val == v {
			return true
		}
	}
	return false
}

func (T *Backend) GameByIDs(ctx context.Context, ids []int, ordering string) ([]game.Game, error) {
	games := []game.Game{}

	for _, id := range ids {
		if id == TestID || in(id, TestIDs) {
			games = append(games, TestGame(id))
		}
	}
	if len(games) == 0 {
		return games, backend.ErrGameNotFound
	}
	return games, nil
}

func (T *Backend) GameByAlias(ctx context.Context, alias string) (game.Game, error) {
	switch alias {
	case TestAlias:
		return TestGame(TestID), nil
	case TestAlias2:
		return TestGame(TestID2), nil
	default:
		return game.Game{}, backend.ErrGameNotFound
	}
}

func (T *Backend) GameByGBID(ctx context.Context, id int) (game.Game, error) {
	switch id {
	case TestGBID:
		return TestGame(TestID), nil
	case TestGBID2:
		return TestGame(TestID2), nil
	case TestGBID3:
		return TestGame(TestID3), nil
	default:
		return game.Game{}, backend.ErrGameNotFound
	}
}

func (T *Backend) AddAlias(ctx context.Context, id int, alias string) error {
	switch id {
	case TestID:
		return nil
	case TestID2:
		return nil
	case TestID3:
		return nil
	default:
		return backend.ErrGameNotFound
	}
}

func (T *Backend) AddGame(ctx context.Context, g game.Game) error {
	return nil
}

func (T *Backend) UpdateGame(ctx context.Context, g game.Game) error {
	return nil
}

func (T *Backend) DeleteGame(ctx context.Context, g game.Game) error {
	return nil
}

func (T *Backend) GetPopularGameIDs(ctx context.Context) ([]int, error) {
	return nil, nil
}

func (T *Backend) UpdateGamePopularity(ctx context.Context, id, popularity int) error {
	return nil
}

func (T *Backend) UpdateProperties(ctx context.Context, id int, properties map[string]string) error {
	return nil
}

func (T *Backend) DeleteProperties(ctx context.Context, id int, properties map[string]string) error {
	return nil
}

func (T *Backend) IsGameBlacklisted(ctx context.Context, id int) bool {
	return id == TestBlacklistedGBID
}

func (T *Backend) UpsertLocalization(ctx context.Context, gameID int, name, locale string) (game.Localizations, error) {
	return TestLocalization(gameID, name, locale), nil
}

func (T *Backend) GetLocalization(ctx context.Context, gameID int, locale string) (game.Localizations, error) {
	return TestLocalization(gameID, TestLocalizedName, locale), nil
}

func (T *Backend) GetLocalizations(ctx context.Context, gameID int) (game.Localizations, error) {
	return TestLocalization(gameID, TestLocalizedName, TestLocale), nil
}

func (T *Backend) GetBulkLocalizations(ctx context.Context, gameIDs []int, locale string) (map[int]game.Localizations, error) {
	localizations := map[int]game.Localizations{}
	for _, g := range gameIDs {
		localizations[g] = TestLocalization(g, TestLocalizedName, locale)
	}
	return localizations, nil
}

func (T *Backend) DeleteLocalization(ctx context.Context, gameID int, locale string) error {
	return nil
}

func (T *Backend) DeleteLocalizations(ctx context.Context, gameID int) error {
	return nil
}

func NewBackend() *Backend {
	return &Backend{}
}
