package backend

import (
	"database/sql"
	"fmt"
	"log"
	"strconv"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/config"
	"code.justin.tv/web/discovery/game"

	"github.com/lib/pq/hstore"

	"golang.org/x/net/context"

	// Use postgres
	_ "github.com/lib/pq"
)

const (
	maxOpenConns         = 5
	maxIdleConns         = 35
	maxQueueSize         = 200
	dbConnTimeout        = 500 * time.Millisecond
	dbRequestTimeout     = 1 * time.Second
	dbMaxConnAge         = 10 * time.Minute
	cronDbRequestTimeout = 5 * time.Second
)

// dbOptions is an array of base options for the database.
var dbOptions = []db.Option{
	db.DriverName("postgres"),
	db.MaxOpenConns(maxOpenConns),
	db.MaxIdleConns(maxIdleConns),
	db.ConnAcquireTimeout(dbConnTimeout),
	db.RequestTimeout(dbRequestTimeout),
	db.MaxConnAge(dbMaxConnAge),
	db.MaxQueueSize(maxQueueSize),
}

// cronDBOptions is an array of base options for the database to be used by crons.
var cronDBOptions = []db.Option{
	db.DriverName("postgres"),
	db.MaxOpenConns(maxOpenConns),
	db.MaxIdleConns(maxIdleConns),
	db.ConnAcquireTimeout(dbConnTimeout),
	db.RequestTimeout(cronDbRequestTimeout),
	db.MaxConnAge(dbMaxConnAge),
	db.MaxQueueSize(maxQueueSize),
}

type dbWrapper struct {
	db db.DB
}

func init() {
	config.Register(map[string]string{
		"pg-master-host": "localhost",
		"pg-master-port": "12150",
		"pg-slave-host":  "localhost",
		"pg-slave-port":  "12151",
		"pg-dbname":      "discovery",
		"pg-user":        "discovery_02",
		"pg-pass":        "",
	})
}

func (db *dbWrapper) Query(ctx context.Context, query dbQuery) (db.Rows, error) {
	return db.db.Query(ctx, query.queryName, query.query, query.args...)
}

func (db *dbWrapper) QueryRow(ctx context.Context, query dbQuery) db.Row {
	return db.db.QueryRow(ctx, query.queryName, query.query, query.args...)
}

func (db *dbWrapper) Exec(ctx context.Context, query dbQuery) (db.Result, error) {
	return db.db.Exec(ctx, query.queryName, query.query, query.args...)
}

func (db *dbWrapper) Close() error {
	return db.db.Close()
}

func (db *dbWrapper) Info() db.DBInfo {
	return db.db.Info()
}

func CronDBOptions() []db.Option {
	cronOpts := cronDBOptions
	masterPort, err := strconv.Atoi(config.MustResolve("pg-master-port"))
	if err != nil {
		log.Printf("PG_MASTER_PORT should be an integer")
	}
	cronOpts = append(
		cronOpts,
		db.DBName(config.MustResolve("pg-dbname")),
		db.User(config.MustResolve("pg-user")),
		db.Password(config.MustResolve("pg-pass")),
		db.Host(config.MustResolve("pg-master-host")),
		db.Port(masterPort),
	)

	return cronOpts
}

func MasterDBOptions() []db.Option {
	masterOpts := dbOptions
	masterPort, err := strconv.Atoi(config.MustResolve("pg-master-port"))
	if err != nil {
		log.Printf("PG_MASTER_PORT should be an integer")
	}
	masterOpts = append(
		masterOpts,
		db.DBName(config.MustResolve("pg-dbname")),
		db.User(config.MustResolve("pg-user")),
		db.Password(config.MustResolve("pg-pass")),
		db.Host(config.MustResolve("pg-master-host")),
		db.Port(masterPort),
	)

	return masterOpts
}

func SlaveDBOptions() []db.Option {
	slaveOpts := dbOptions
	slavePort, err := strconv.Atoi(config.MustResolve("pg-slave-port"))
	if err != nil {
		log.Printf("PG_SLAVE_PORT should be an integer")
	}

	slaveOpts = append(
		slaveOpts,
		db.DBName(config.MustResolve("pg-dbname")),
		db.User(config.MustResolve("pg-user")),
		db.Password(config.MustResolve("pg-pass")),
		db.Host(config.MustResolve("pg-slave-host")),
		db.Port(slavePort),
	)

	return slaveOpts
}

func mapToHstore(m map[string]string) hstore.Hstore {
	store := hstore.Hstore{
		Map: make(map[string]sql.NullString),
	}

	for k, v := range m {
		store.Map[k] = sql.NullString{String: v, Valid: true}
	}

	return store
}

// ScanGame converts a database row to a Game struct.
// The caller must take care of calling row.Next()
func ScanGame(row db.Rows) (game.Game, error) {
	var g game.Game
	var img, properties dbHstore
	var gbID, popularity sql.NullInt64
	if err := row.Scan(&g.ID,
		&gbID,
		&g.Name,
		&img,
		&popularity,
		&properties); err != nil {
		return g, err
	}
	if gbID.Valid {
		// TODO: make sure that giantbomb id of 0 is handled properly
		g.GiantbombID = int(gbID.Int64)
	}
	if popularity.Valid {
		g.Popularity = int(popularity.Int64)
	}
	g.Images = img.ToMap()
	g.Properties = properties.ToMap()
	g.BuildArt()
	return g, nil
}

// ScanLiveGame converts a database row to a LiveGame struct.
// The caller must take care of calling row.Next()
func ScanLiveGame(row db.Rows) (game.LiveGame, error) {
	g := game.LiveGame{Game: game.Game{}}
	var img, properties dbHstore
	var gbID, popularity sql.NullInt64
	if err := row.Scan(
		&g.Viewers,
		&g.Channels,
		&g.Game.ID,
		&gbID,
		&g.Game.Name,
		&img,
		&popularity,
		&properties); err != nil {
		return g, err
	}
	if gbID.Valid {
		// TODO: make sure that giantbomb id of 0 is handled properly
		g.Game.GiantbombID = int(gbID.Int64)
	}
	if popularity.Valid {
		g.Game.Popularity = int(popularity.Int64)
	}
	g.Game.Images = img.ToMap()
	g.Game.Properties = properties.ToMap()
	g.Game.BuildArt()
	return g, nil
}

// ScanLocalization converts a database row to a Localization struct.
// The caller must take care of calling row.Next()
func ScanLocalization(row db.Rows) (game.LocalizationData, error) {
	var l game.LocalizationData
	if err := row.Scan(&l.GameID,
		&l.Name,
		&l.Locale); err != nil {
		return l, err
	}
	return l, nil
}

type dbHstore struct {
	hstore.Hstore
}

// ToMap converts dbHstore to a raw map, removing nullStrings and such
func (h *dbHstore) ToMap() map[string]string {
	m := make(map[string]string)
	for k, v := range h.Map {
		if v.Valid {
			val, _ := v.Value()
			m[k] = val.(string)
		}
	}
	return m
}

// OpenDB creates a sql.DB. The caller is responsible for closing the connection.
func OpenDB(name, user, password, host string, port int) (*sql.DB, error) {
	pgConnect := fmt.Sprintf("dbname=%s user=%s password=%s host=%s port=%d sslmode=disable",
		name, user, password, host, port)

	db, err := sql.Open("postgres", pgConnect)
	if err != nil {
		return nil, fmt.Errorf("backend: got err while connecting to db (%v)", err)
	}

	// Connections aren't checked till a query is performed, so make sure we do so
	if db.Ping() != nil {
		return nil, fmt.Errorf("backend: unable to connect to db (%v)", err)
	}

	return db, nil
}
