package backend

import (
	"strconv"
	"strings"

	"github.com/lib/pq/hstore"
)

const sqlHeader = "-- discovery backend/db_queries.go\n"

func testConnectionQuery() string {
	query := `SELECT now()`
	return query
}

type dbQuery struct {
	queryName string
	query     string
	args      []interface{}
}

func suggestGamesQuery(term string) dbQuery {
	return dbQuery{
		queryName: "suggest",
		query: sqlHeader + `SELECT * FROM games
WHERE id IN
	(SELECT id FROM games
	WHERE lower(name) LIKE $1
	UNION
	SELECT game_id FROM game_aliases
	WHERE lower(alias) LIKE $1)
ORDER BY popularity DESC NULLS LAST
LIMIT 50`,
		args: []interface{}{strings.ToLower(term) + "%"},
	}
}

func recentVersion(version int64) dbQuery {
	return dbQuery{
		queryName: "recent_version",
		query: `SELECT COALESCE((SELECT version FROM top_games WHERE version=$1 LIMIT 1), ` +
			`(SELECT MAX(version) FROM top_games))`,
		args: []interface{}{version},
	}
}
func totalTopGames(version int64) dbQuery {
	return dbQuery{
		queryName: "count_games",
		query: `SELECT COUNT(*) FROM top_games tg ` +
			`WHERE version=COALESCE((SELECT version FROM top_games WHERE version=$1 LIMIT 1), (SELECT MAX(version) FROM top_games))`,
		args: []interface{}{version},
	}
}

func topGames(version int64, limit, offset int) dbQuery {
	return dbQuery{
		queryName: "top_games",
		query: `SELECT tg.viewers, tg.channels, g.id, g.giantbomb_id, g.name, g.images, g.popularity, g.properties ` +
			`FROM top_games tg JOIN games g ON tg.game_id=g.id ` +
			`WHERE tg.version=COALESCE((SELECT version FROM top_games WHERE version=$1 LIMIT 1), (SELECT MAX(version) FROM top_games)) ` +
			`ORDER BY tg.viewers DESC, g.id LIMIT $2 OFFSET $3`,
		args: []interface{}{version, limit, offset},
	}
}

func topGamesByID(version int64, ids []int, limit, offset int) dbQuery {
	idList := buildIDList(ids)
	return dbQuery{
		queryName: "top_games_by_ids",
		query: `SELECT tg.viewers, tg.channels, g.id, g.giantbomb_id, g.name, g.images, g.popularity, g.properties ` +
			`FROM top_games tg JOIN games g ON tg.game_id=g.id ` +
			`WHERE tg.version=COALESCE((SELECT version FROM top_games WHERE version=$1 LIMIT 1), (SELECT MAX(version) FROM top_games)) ` +
			`AND tg.game_id IN ` + idList +
			` ORDER BY tg.viewers DESC, g.id LIMIT $2 OFFSET $3`,
		args: []interface{}{version, limit, offset},
	}
}

func buildIDList(ids []int) string {
	idList := "("
	for i, id := range ids {
		idList += strconv.Itoa(id)
		if i < len(ids)-1 {
			idList += ","
		}
	}
	idList += ")"
	return idList
}

func searchID(id int) dbQuery {
	return dbQuery{
		queryName: "by_id",
		query:     sqlHeader + "SELECT * FROM games WHERE id = $1 LIMIT 1",
		args:      []interface{}{id},
	}
}

func searchIDs(ids []int, ordering string) dbQuery {
	query := sqlHeader + "SELECT * FROM games WHERE id IN ("
	for i, id := range ids {
		query += strconv.Itoa(id)
		if i < len(ids)-1 {
			query += ","
		}
	}
	query += ")"
	if len(ordering) > 0 && ordering != DefaultOrdering {
		query += " ORDER BY " + ordering
	}

	return dbQuery{
		queryName: "by_ids",
		query:     query,
		args:      []interface{}{},
	}
}

func searchName(name string) dbQuery {
	return dbQuery{
		queryName: "by_name",
		query:     sqlHeader + "SELECT * FROM games WHERE lower(name) = $1 LIMIT 1",
		args:      []interface{}{strings.ToLower(name)},
	}
}

func searchGBID(id int) dbQuery {
	return dbQuery{
		queryName: "by_giantbomb_id",
		query:     sqlHeader + "SELECT * FROM games WHERE giantbomb_id = $1 LIMIT 1",
		args:      []interface{}{id},
	}
}

func searchAlias(alias string) dbQuery {
	return dbQuery{
		queryName: "by_alias",
		query:     sqlHeader + "SELECT * FROM games WHERE id IN (SELECT game_id FROM game_aliases WHERE lower(alias) = $1) ORDER BY popularity DESC LIMIT 1",
		args:      []interface{}{strings.ToLower(alias)},
	}
}

func searchNameOrAlias(term string) dbQuery {
	return dbQuery{
		queryName: "by_name_alias",
		query: sqlHeader + `SELECT * FROM games
WHERE id IN
	(SELECT id FROM games
	WHERE lower(name) = $1
	UNION
	SELECT game_id FROM game_aliases
	WHERE lower(alias) = $1)
LIMIT 1`,
		args: []interface{}{strings.ToLower(term)},
	}
}

func updatePopularityQuery(id, popularity int) dbQuery {
	return dbQuery{
		queryName: "update_popularity",
		query: sqlHeader + `UPDATE games
SET popularity = $2
WHERE id = $1`,
		args: []interface{}{id, popularity},
	}
}

func popularGamesQuery() dbQuery {
	return dbQuery{
		queryName: "popular_games",
		query: sqlHeader + `SELECT id FROM games
WHERE popularity > 0`,
		args: []interface{}{},
	}
}

func gameIDMappingQuery() dbQuery {
	return dbQuery{
		queryName: "game_id_mapping",
		query: sqlHeader + `SELECT id, name, COALESCE(giantbomb_id, 0)
		FROM games
		ORDER BY id ASC`,
		args: []interface{}{},
	}
}

func allAliasesQuery() dbQuery {
	return dbQuery{
		queryName: "all_aliases_mapping",
		query: sqlHeader + `SELECT game_id, alias
		FROM game_aliases`,
		args: []interface{}{},
	}
}

func aliasesByGameIDQuery(id int) dbQuery {
	return dbQuery{
		queryName: "aliases",
		query:     sqlHeader + `SELECT alias FROM game_aliases WHERE game_id = $1`,
		args:      []interface{}{id},
	}
}

func genresByGameIDQuery(id int) dbQuery {
	return dbQuery{
		queryName: "genres",
		query: sqlHeader + `SELECT gen.name FROM genres gen, games g, game_genre gg
WHERE gg.genre_id = gen.id
AND gg.game_id = g.id
AND g.id = $1`,
		args: []interface{}{id},
	}
}

func insertAliasQuery(id int, alias string) dbQuery {
	return dbQuery{
		queryName: "add_alias",
		query:     sqlHeader + `INSERT INTO game_aliases (alias, game_id) VALUES ($1, $2)`,
		args:      []interface{}{alias, id},
	}
}

func insertGameQuery(
	giantbombID, popularity int,
	name string,
	images, properties hstore.Hstore,
) dbQuery {
	return dbQuery{
		queryName: "insert_game",
		query: sqlHeader + `INSERT INTO games
	(giantbomb_id, name, popularity, images, properties)
	VALUES ($1, $2, $3, $4::hstore, $5::hstore)`,
		args: []interface{}{giantbombID, name, popularity, images, properties},
	}
}

func insertGameNoGBID(
	popularity int,
	name string,
	images, properties hstore.Hstore,
) dbQuery {
	return dbQuery{
		queryName: "insert_game_no_gb",
		query: sqlHeader + `INSERT INTO games
	(name, popularity, images, properties)
	VALUES ($1, $2, $3::hstore, $4::hstore)`,
		args: []interface{}{name, popularity, images, properties},
	}
}

func updateGameQuery(
	id, giantbombID, popularity int,
	name string,
	images, properties hstore.Hstore,
) dbQuery {
	return dbQuery{
		queryName: "update_game",
		query: sqlHeader + `UPDATE games SET
giantbomb_id=$2,
name=$4,
popularity=$3,
images=$5::hstore,
properties=$6::hstore
WHERE id=$1`,
		args: []interface{}{id, giantbombID, popularity, name, images, properties},
	}
}

func deleteGameQuery(id int) dbQuery {
	return dbQuery{
		queryName: "delete_game",
		query:     sqlHeader + `DELETE FROM games WHERE id=$1`,
		args:      []interface{}{id},
	}
}

func updatePropertiesQuery(id int, properties hstore.Hstore) dbQuery {
	return dbQuery{
		queryName: "update_properties",
		query: sqlHeader + `UPDATE games
SET properties = properties || $2::hstore
WHERE id=$1`,
		args: []interface{}{id, properties},
	}
}

func deletePropertiesQuery(id int, toDelete map[string]string) dbQuery {
	query := sqlHeader + `UPDATE games
SET properties = properties - ARRAY[`

	for key := range toDelete {
		query += `'` + key + `',`
	}
	// remove trailing comma
	query = query[:len(query)-1]

	query += `] WHERE id=$1`
	return dbQuery{
		queryName: "delete_properties",
		query:     query,
		args:      []interface{}{id},
	}
}

func blacklistGameQuery(giantbombID int) dbQuery {
	return dbQuery{
		queryName: "blacklist_game",
		query:     sqlHeader + `INSERT INTO giantbomb_blacklist (giantbomb_id) VALUES ($1)`,
		args:      []interface{}{giantbombID},
	}
}

func isBlacklistQuery(id int) dbQuery {
	return dbQuery{
		queryName: "is_blacklisted",
		query:     sqlHeader + "SELECT exists(SELECT 1 FROM giantbomb_blacklist WHERE giantbomb_id=$1)",
		args:      []interface{}{id},
	}
}

func upsertLocalizationQuery(
	gameID int,
	name, locale string,
) dbQuery {
	return dbQuery{
		queryName: "upsert_localization",
		query: sqlHeader + `INSERT INTO localizations
		(game_id, name, locale)
		VALUES ($1, $2, $3)
		ON CONFLICT (game_id, locale)
		DO UPDATE SET name=$2
		WHERE (localizations.game_id, localizations.locale) = ($1, $3)
		RETURNING game_id, name, locale`,
		args: []interface{}{gameID, name, locale},
	}
}

func getLocalizationsQuery(gameID int) dbQuery {
	return dbQuery{
		queryName: "get_localizations",
		query:     sqlHeader + "SELECT game_id, name, locale FROM localizations WHERE game_id=$1",
		args:      []interface{}{gameID},
	}
}

func getLocalizationQuery(gameID int, locale string) dbQuery {
	return dbQuery{
		queryName: "get_localization",
		query:     sqlHeader + "SELECT game_id, name, locale FROM localizations WHERE game_id=$1 AND locale=$2",
		args:      []interface{}{gameID, locale},
	}
}

func getBulkLocalizationsQuery(ids []int, locale string) dbQuery {
	query := sqlHeader + "SELECT game_id, name, locale FROM localizations WHERE locale=$1 AND game_id IN ("
	for i, id := range ids {
		query += strconv.Itoa(id)
		if i < len(ids)-1 {
			query += ","
		}
	}
	query += ")"

	return dbQuery{
		queryName: "bulk_get_localizations",
		query:     query,
		args:      []interface{}{locale},
	}
}

func deleteLocalizationsQuery(gameID int) dbQuery {
	return dbQuery{
		queryName: "delete_localizations",
		query:     sqlHeader + "DELETE FROM localizations WHERE game_id=$1",
		args:      []interface{}{gameID},
	}
}

func deleteLocalizationQuery(gameID int, locale string) dbQuery {
	return dbQuery{
		queryName: "delete_localization",
		query:     sqlHeader + "DELETE FROM localizations WHERE game_id=$1 AND locale=$2",
		args:      []interface{}{gameID, locale},
	}
}
