package backend

import (
	"testing"

	"github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"
)

func TestScanOneGame(t *testing.T) {
	mockDB, mock := NewMockDB()

	columns := []string{"id", "giantbomb_id", "name", "images", "popularity", "properites"}

	mock.ExpectQuery("SELECT (.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows(columns).FromCSVString(
			`1,2,test_game,small=>hi,3,
		a,2,test_game,small=>hi,3,`))

	rows, err := mockDB.Query("SELECT * FROM games")
	Convey("query should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	rows.Next()
	g, err := ScanGame(rows)
	Convey("game reading should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should have right basic data", t, func() {
		So(g.ID, ShouldEqual, 1)
		So(g.GiantbombID, ShouldEqual, 2)
		So(g.Name, ShouldEqual, "test_game")
	})

	Convey("should have right map data", t, func() {
		So(g.Images, ShouldResemble, map[string]string{"small": "hi"})
		So(g.Properties, ShouldResemble, map[string]string{})
	})

	Convey("should have art", t, func() {
		So(g.BoxArt, ShouldNotBeEmpty)
		So(g.LogoArt, ShouldNotBeEmpty)
	})

	rows.Next()
	g, err = ScanGame(rows)
	Convey("should error for wrong type", t, func() {
		So(err, ShouldNotBeNil)
	})
}

func TestScanOneLocalization(t *testing.T) {
	mockDB, mock := NewMockDB()

	columns := []string{"game_id", "name", "locale"}

	mock.ExpectQuery("SELECT (.+) FROM (.+)").
		WillReturnRows(sqlmock.NewRows(columns).FromCSVString(
			`1,test_localized_name,test_locale
		a,test_localized_name,test_locale`))

	rows, err := mockDB.Query("SELECT (game_id, name, locale) FROM localizations")
	Convey("query should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	rows.Next()
	l, err := ScanLocalization(rows)
	Convey("localization reading should not have an error", t, func() {
		So(err, ShouldBeNil)
	})

	Convey("should have right basic data", t, func() {
		So(l.GameID, ShouldEqual, 1)
		So(l.Name, ShouldEqual, "test_localized_name")
		So(l.Locale, ShouldEqual, "test_locale")
	})

	rows.Next()
	l, err = ScanLocalization(rows)
	Convey("should error for wrong type", t, func() {
		So(err, ShouldNotBeNil)
	})
}
