package discovery

import (
	"errors"
	"fmt"
	"net/url"
	"strings"

	"strconv"

	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "discovery"
)

type Client interface {
	Get(ctx context.Context, id, locale string, reqOpts *twitchclient.ReqOpts) (*LocalizedGame, error)
	GetAll(ctx context.Context, ids []string, locale string, limit, offset int, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error)
	GetByName(ctx context.Context, name, locale string, reqOpts *twitchclient.ReqOpts) (*LocalizedGame, error)
	BulkGetGamesByIDs(ctx context.Context, ids []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error)
	BulkGetGamesByNames(ctx context.Context, names []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error)
	GetAllAliasesByID(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) ([]string, error)
	GetAllAliasesByName(ctx context.Context, name string, reqOpts *twitchclient.ReqOpts) ([]string, error)
	Suggest(ctx context.Context, term, locale string, params *SuggestParams, reqOpts *twitchclient.ReqOpts) ([]LocalizedGame, error)
	GetLocalizationsByID(ctx context.Context, gameID, locale string, reqOpts *twitchclient.ReqOpts) (*Localizations, error)
	GetLocalizationsByName(ctx context.Context, gameName, locale string, reqOpts *twitchclient.ReqOpts) (*Localizations, error)
	BulkGetLocalizationsByIDs(ctx context.Context, gameIDs []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*Localizations, error)
	BulkGetLocalizationsByNames(ctx context.Context, gameNames []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*Localizations, error)
	AddLocalizationByID(ctx context.Context, gameID, locale, localizedName string, reqOpts *twitchclient.ReqOpts) (*Localizations, error)
	AddLocalizationByName(ctx context.Context, gameName, locale, localizedName string, reqOpts *twitchclient.ReqOpts) (*Localizations, error)
	DeleteLocalizationByID(ctx context.Context, gameID, locale string, reqOpts *twitchclient.ReqOpts) error
	DeleteLocalizationByName(ctx context.Context, gameName, locale string, reqOpts *twitchclient.ReqOpts) error
	DeleteAllLocalizationsByID(ctx context.Context, gameID string, reqOpts *twitchclient.ReqOpts) error
	DeleteAllLocalizationsByName(ctx context.Context, gameName string, reqOpts *twitchclient.ReqOpts) error
}

type client struct {
	twitchclient.Client
}

type bulkResult struct {
	key  string
	data interface{}
	err  error
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) Get(ctx context.Context, id, locale string, reqOpts *twitchclient.ReqOpts) (*LocalizedGame, error) {
	return c.get(ctx, "id", id, locale, "service.discovery.get", reqOpts)
}

func (c *client) GetByName(ctx context.Context, name, locale string, reqOpts *twitchclient.ReqOpts) (*LocalizedGame, error) {
	return c.get(ctx, "name", name, locale, "service.discovery.get_by_name", reqOpts)
}

func (c *client) Suggest(ctx context.Context, term, locale string, params *SuggestParams, reqOpts *twitchclient.ReqOpts) ([]LocalizedGame, error) {
	query := url.Values{
		"term":   {term},
		"locale": {locale},
	}
	if params != nil {
		if params.Live {
			query.Add("live", "true")
		}
	}

	req, err := c.NewRequest("GET", "/suggest?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.discovery.suggest",
		StatSampleRate: defaultStatSampleRate,
	})

	var data []LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) GetAll(ctx context.Context, ids []string, locale string, limit, offset int, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error) {
	query := url.Values{
		"id":     {strings.Join(ids, ",")},
		"limit":  {strconv.Itoa(limit)},
		"offset": {strconv.Itoa(offset)},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/game/list?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.discovery.get_all",
		StatSampleRate: defaultStatSampleRate,
	})

	var data []*LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	resp := make(map[string]*LocalizedGame)
	for _, game := range data {
		resp[strconv.Itoa(game.Game.ID)] = game
	}
	return resp, nil
}

func (c *client) BulkGetGamesByIDs(ctx context.Context, ids []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error) {
	return c.getBulkGames(ctx, "id", ids, locale, "service.discovery.get_bulk_games_by_ids", reqOpts)
}

func (c *client) BulkGetGamesByNames(ctx context.Context, names []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error) {
	return c.getBulkGames(ctx, "name", names, locale, "service.discovery.get_bulk_games_by_names", reqOpts)
}

func (c *client) getBulkGames(ctx context.Context, key string, values []string, locale, stat string, reqOpts *twitchclient.ReqOpts) (map[string]*LocalizedGame, error) {
	values = getUniques(values)

	n := len(values)
	var resultsC = make(chan *bulkResult, n)

	for _, v := range values {
		go func(v string) {
			// Make requests concurrently to minimize request latency.
			defer func() {
				// Cleanup possible panics so they don't kill the application.
				if err := recover(); err != nil {
					resultsC <- &bulkResult{v, nil, fmt.Errorf("panic in getBulkGames=%v", err)}
				}
			}()
			game, err := c.get(ctx, key, v, locale, stat, reqOpts)
			resultsC <- &bulkResult{v, game, err}
		}(v)
	}

	bulkGames := map[string]*LocalizedGame{}
	for i := 0; i < n; i++ {
		r := <-resultsC
		if r.err != nil {
			if tErr, ok := r.err.(*twitchclient.Error); !ok || tErr.StatusCode >= 500 {
				return nil, r.err
			}
			continue
		}
		game, ok := r.data.(*LocalizedGame)
		if !ok {
			return nil, errors.New("unable to parse LocalizedGame")
		}
		bulkGames[r.key] = game
	}

	return bulkGames, nil
}

func (c *client) GetAllAliasesByID(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	return c.getAllAliases(ctx, "id", id, "service.discovery.get_all_aliases_by_id", reqOpts)
}

func (c *client) GetAllAliasesByName(ctx context.Context, name string, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	return c.getAllAliases(ctx, "name", name, "service.discovery.get_all_aliases_by_name", reqOpts)
}

func (c *client) getAllAliases(ctx context.Context, key, value, statName string, reqOpts *twitchclient.ReqOpts) ([]string, error) {
	query := url.Values{
		key: {value},
	}

	req, err := c.NewRequest("GET", "/aliases?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})

	var data Aliases
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return data.Aliases, err
}

func (c *client) GetLocalizationsByID(ctx context.Context, gameID, locale string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	return c.getLocalizations(ctx, "game_id", gameID, locale, "service.discovery.get_localizations_by_id", reqOpts)
}

func (c *client) GetLocalizationsByName(ctx context.Context, gameName, locale string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	return c.getLocalizations(ctx, "game_name", gameName, locale, "service.discovery.get_localizations_by_name", reqOpts)
}

func (c *client) BulkGetLocalizationsByIDs(ctx context.Context, gameIDs []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*Localizations, error) {
	return c.getBulkLocalizations(ctx, "game_id", gameIDs, locale, "service.discovery.get_bulk_localizations_by_ids", reqOpts)
}

func (c *client) BulkGetLocalizationsByNames(ctx context.Context, gameNames []string, locale string, reqOpts *twitchclient.ReqOpts) (map[string]*Localizations, error) {
	return c.getBulkLocalizations(ctx, "game_name", gameNames, locale, "service.discovery.get_bulk_localizations_by_names", reqOpts)
}

func (c *client) AddLocalizationByID(ctx context.Context, gameID, locale, localizedName string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	return c.addLocalization(ctx, "game_id", gameID, locale, localizedName, "service.discovery.add_localization_by_id", reqOpts)
}

func (c *client) AddLocalizationByName(ctx context.Context, gameName, locale, localizedName string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	return c.addLocalization(ctx, "game_name", gameName, locale, localizedName, "service.discovery.add_localization_by_name", reqOpts)
}

func (c *client) DeleteLocalizationByID(ctx context.Context, gameID, locale string, reqOpts *twitchclient.ReqOpts) error {
	return c.deleteLocalization(ctx, "game_id", gameID, locale, "service.discovery.delete_localizations_by_id", reqOpts)
}

func (c *client) DeleteLocalizationByName(ctx context.Context, gameName, locale string, reqOpts *twitchclient.ReqOpts) error {
	return c.deleteLocalization(ctx, "game_name", gameName, locale, "service.discovery.delete_localizations_by_name", reqOpts)
}

func (c *client) DeleteAllLocalizationsByID(ctx context.Context, gameID string, reqOpts *twitchclient.ReqOpts) error {
	return c.deleteAllLocalizations(ctx, "game_id", gameID, "service.discovery.delete_all_localizations_by_id", reqOpts)
}

func (c *client) DeleteAllLocalizationsByName(ctx context.Context, gameName string, reqOpts *twitchclient.ReqOpts) error {
	return c.deleteAllLocalizations(ctx, "game_name", gameName, "service.discovery.delete_all_localizations_by_name", reqOpts)
}

func (c *client) get(ctx context.Context, key, value, locale, stat string, reqOpts *twitchclient.ReqOpts) (*LocalizedGame, error) {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/game?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) getLocalizations(ctx context.Context, key, value, locale, stat string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/localizations?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *Localizations
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) getBulkLocalizations(ctx context.Context, key string, values []string, locale, stat string, reqOpts *twitchclient.ReqOpts) (map[string]*Localizations, error) {
	values = getUniques(values)

	n := len(values)
	var resultsC = make(chan *bulkResult, n)

	for _, v := range values {
		go func(v string) {
			// Make requests concurrently to minimize request latency.
			defer func() {
				// Cleanup possible panics so they don't kill the application.
				if err := recover(); err != nil {
					resultsC <- &bulkResult{v, nil, fmt.Errorf("panic in getBulkLocalizatons=%v", err)}
				}
			}()
			localizations, err := c.getLocalizations(ctx, key, v, locale, stat, reqOpts)
			resultsC <- &bulkResult{v, localizations, err}
		}(v)
	}

	bulkLocalizations := map[string]*Localizations{}
	for i := 0; i < n; i++ {
		r := <-resultsC
		if r.err != nil {
			if tErr, ok := r.err.(*twitchclient.Error); !ok || tErr.StatusCode >= 500 {
				return nil, r.err
			}
			continue
		}
		localizations, ok := r.data.(*Localizations)
		if !ok {
			return nil, errors.New("unable to parse Localizations")
		}
		bulkLocalizations[r.key] = localizations
	}

	return bulkLocalizations, nil
}

func (c *client) addLocalization(ctx context.Context, key, value, locale, localizedName, stat string, reqOpts *twitchclient.ReqOpts) (*Localizations, error) {
	form := url.Values{
		key:              {value},
		"locale":         {locale},
		"localized_name": {localizedName},
	}

	req, err := c.NewRequest("PUT", "/localizations", strings.NewReader(form.Encode()))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *Localizations
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) deleteLocalization(ctx context.Context, key, value, locale, stat string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("DELETE", "/localizations?"+query.Encode(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = c.DoJSON(ctx, nil, req, combinedReqOpts)
	return err
}

func (c *client) deleteAllLocalizations(ctx context.Context, key, value, stat string, reqOpts *twitchclient.ReqOpts) error {
	query := url.Values{
		key: {value},
	}

	req, err := c.NewRequest("DELETE", "/localizations/all?"+query.Encode(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = c.DoJSON(ctx, nil, req, combinedReqOpts)
	return err
}

// Returns unique strings from slice
func getUniques(s []string) []string {
	u := []string{}
	seen := make(map[string]struct{}, len(s))
	for _, v := range s {
		if _, ok := seen[v]; ok {
			continue
		}
		seen[v] = struct{}{}
		u = append(u, v)
	}
	return u
}
