package discovery

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"

	. "github.com/smartystreets/goconvey/convey"
)

const (
	bulkGetGamesIDsResponseFile   = "testdata/bulk_get_games_ids_success.json"
	bulkGetGamesNamesResponseFile = "testdata/bulk_get_games_ids_success.json"
	gameLocaleResponseFile        = "testdata/game_success_locale.json"
	suggestLocaleResponseFile     = "testdata/suggest_success_locale.json"
	gameListLocaleResponseFile    = "testdata/game_list_success_locale.json"
	gameNoLocaleResponseFile      = "testdata/game_success_no_locale.json"
	suggestNoLocaleResponseFile   = "testdata/suggest_success_no_locale.json"
	gameListNoLocaleResponseFile  = "testdata/game_list_success_no_locale.json"
	localizationsResponseFile     = "testdata/localizations_success.json"

	testGameID        = 19813
	testGameIDString  = "19813"
	testGameName      = "Metroid"
	testEmptyLocale   = ""
	testLocale        = "ja-jp"
	testLocalizedName = "メトロイド"

	testListGameID        = 7595
	testListGameIDString  = "7595"
	testListGameName      = "Super Metroid"
	testListLocalizedName = "スーパーメトロイド"
)

var testLocalizations = &Localizations{
	GameID:         testGameID,
	LocalizedNames: map[string]string{testLocale: testLocalizedName},
}

var testListLocalizations = &Localizations{
	GameID:         testListGameID,
	LocalizedNames: map[string]string{testLocale: testListLocalizedName},
}

func readTestFile(name string) []byte {
	response, err := ioutil.ReadFile(name)
	if err != nil {
		log.Fatalf("Error reading file %s: %v", name, err)
	}
	return response
}

func TestGet(t *testing.T) {
	Convey("when calling Get by id", t, func() {
		Convey("with locale", func() {
			gameResponse := readTestFile(gameLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						fmt.Fprintln(w, string(gameResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				game, err := d.Get(context.Background(), testGameIDString, testLocale, nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(game.Name, ShouldEqual, testGameName)
					So(game.Locale, ShouldEqual, "ja-jp")
					So(game.LocalizedName, ShouldEqual, testLocalizedName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Get(context.Background(), testGameIDString, testLocale, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("without locale", func() {
			gameResponse := readTestFile(gameNoLocaleResponseFile)
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						fmt.Fprintln(w, string(gameResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				game, err := d.Get(context.Background(), testGameIDString, "", nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(game.Name, ShouldEqual, testGameName)
					So(game.Locale, ShouldEqual, "")
					So(game.LocalizedName, ShouldEqual, testGameName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Get(context.Background(), testGameIDString, "", nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})

	Convey("when calling Get by name", t, func() {
		Convey("with locale", func() {
			gameResponse := readTestFile(gameLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						fmt.Fprintln(w, string(gameResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				game, err := d.Get(context.Background(), testGameName, testLocale, nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(game.Name, ShouldEqual, testGameName)
					So(game.Locale, ShouldEqual, "ja-jp")
					So(game.LocalizedName, ShouldEqual, testLocalizedName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Get(context.Background(), testGameName, testLocale, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("without locale", func() {
			gameResponse := readTestFile(gameNoLocaleResponseFile)
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						fmt.Fprintln(w, string(gameResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				game, err := d.Get(context.Background(), testGameName, "", nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(game.Name, ShouldEqual, testGameName)
					So(game.Locale, ShouldEqual, "")
					So(game.LocalizedName, ShouldEqual, testGameName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Get(context.Background(), testGameName, "", nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestSuggest(t *testing.T) {
	Convey("when calling Suggest", t, func() {
		Convey("with locale", func() {
			suggestResponse := readTestFile(suggestLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/suggest" {
						fmt.Fprintln(w, string(suggestResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				games, err := d.Suggest(context.Background(), testGameName, testLocale, nil, nil)

				Convey("should return a list of games", func() {
					So(err, ShouldBeNil)
					So(len(games), ShouldEqual, 2)
					So(games[0].Name, ShouldEqual, testListGameName)
					So(games[0].Locale, ShouldEqual, "ja-jp")
					So(games[0].LocalizedName, ShouldEqual, testListLocalizedName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/suggest" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Suggest(context.Background(), testGameName, testLocale, nil, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("without locale", func() {
			suggestResponse := readTestFile(suggestNoLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/suggest" {
						fmt.Fprintln(w, string(suggestResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				games, err := d.Suggest(context.Background(), testGameName, "", nil, nil)

				Convey("should return a list of games", func() {
					So(err, ShouldBeNil)
					So(len(games), ShouldEqual, 2)
					So(games[0].Name, ShouldEqual, testListGameName)
					So(games[0].Locale, ShouldEqual, "")
					So(games[0].LocalizedName, ShouldEqual, testListGameName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/suggest" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.Suggest(context.Background(), testGameName, "", nil, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestGetAll(t *testing.T) {
	Convey("when calling GetAll", t, func() {
		Convey("with locale", func() {
			gameListResponse := readTestFile(gameListLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game/list" {
						fmt.Fprintln(w, string(gameListResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				games, err := d.GetAll(context.Background(), nil, testLocale, 100, 0, nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(len(games), ShouldEqual, 2)
					So(games[testListGameIDString].Name, ShouldEqual, testListGameName)
					So(games[testListGameIDString].Locale, ShouldEqual, "ja-jp")
					So(games[testListGameIDString].LocalizedName, ShouldEqual, testListLocalizedName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game/list" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetAll(context.Background(), nil, testLocale, 100, 0, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("without locale", func() {
			gameListResponse := readTestFile(gameListNoLocaleResponseFile)

			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game/list" {
						fmt.Fprintln(w, string(gameListResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				games, err := d.GetAll(context.Background(), nil, "", 100, 0, nil)

				Convey("should return a game", func() {
					So(err, ShouldBeNil)
					So(len(games), ShouldEqual, 2)
					So(games[testListGameIDString].Name, ShouldEqual, testListGameName)
					So(games[testListGameIDString].Locale, ShouldEqual, "")
					So(games[testListGameIDString].LocalizedName, ShouldEqual, testListGameName)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/game/list" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetAll(context.Background(), nil, "", 100, 0, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestBulkGetGamesByIDs(t *testing.T) {
	Convey("when calling BulkGetGamesByIDs", t, func() {
		bulkGamesResponse := make(map[string]*LocalizedGame)
		bulkGamesResponseBytes := readTestFile(bulkGetGamesIDsResponseFile)

		_ = json.Unmarshal(bulkGamesResponseBytes, &bulkGamesResponse)

		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameID := r.URL.Query().Get("id")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameID])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByIDs(context.Background(), []string{testGameIDString, testListGameIDString}, "", nil)

			Convey("should return bulk games", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 2)
				So(bulkGames[testGameIDString], ShouldResemble, bulkGamesResponse[testGameIDString])
				So(bulkGames[testListGameIDString], ShouldResemble, bulkGamesResponse[testListGameIDString])
			})
		})
		Convey("handling duplicate slice entries", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameID := r.URL.Query().Get("id")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameID])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByIDs(context.Background(), []string{testGameIDString, testGameIDString, testListGameIDString}, "", nil)

			Convey("should return bulk games", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 2)
				So(bulkGames[testGameIDString], ShouldResemble, bulkGamesResponse[testGameIDString])
				So(bulkGames[testListGameIDString], ShouldResemble, bulkGamesResponse[testListGameIDString])
			})
		})
		Convey("handling empty slice", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameID := r.URL.Query().Get("id")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameID])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByIDs(context.Background(), []string{}, "", nil)

			Convey("should return empty result", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 0)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.BulkGetGamesByIDs(context.Background(), []string{testGameIDString, testListGameIDString}, "", nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestBulkGetGamesByNames(t *testing.T) {
	Convey("when calling BulkGetGamesByNames", t, func() {
		bulkGamesResponse := make(map[string]*LocalizedGame)
		bulkGamesResponseBytes := readTestFile(bulkGetGamesNamesResponseFile)

		_ = json.Unmarshal(bulkGamesResponseBytes, &bulkGamesResponse)

		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameName := r.URL.Query().Get("name")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameName])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByNames(context.Background(), []string{testGameName, testListGameName}, "", nil)

			Convey("should return bulk games", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 2)
				So(bulkGames[testGameName], ShouldResemble, bulkGamesResponse[testGameName])
				So(bulkGames[testListGameName], ShouldResemble, bulkGamesResponse[testListGameName])
			})
		})
		Convey("handling duplicate slice entries", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameID := r.URL.Query().Get("id")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameID])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByNames(context.Background(), []string{testGameName, testGameName, testListGameName}, "", nil)

			Convey("should return bulk games", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 2)
				So(bulkGames[testGameName], ShouldResemble, bulkGamesResponse[testGameName])
				So(bulkGames[testListGameName], ShouldResemble, bulkGamesResponse[testListGameName])
			})
		})
		Convey("handling empty slice", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					urlGameID := r.URL.Query().Get("id")
					gameResponse, _ := json.Marshal(bulkGamesResponse[urlGameID])
					fmt.Fprintln(w, string(gameResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkGames, err := d.BulkGetGamesByNames(context.Background(), []string{}, "", nil)

			Convey("should return empty result", func() {
				So(err, ShouldBeNil)
				So(len(bulkGames), ShouldEqual, 0)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/game" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.BulkGetGamesByNames(context.Background(), []string{testGameName, testListGameName}, "", nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetAllAliasesByID(t *testing.T) {
	Convey("when calling GetAllAliasesByID", t, func() {
		aliasesResponse, _ := json.Marshal(Aliases{
			Aliases: []string{testGameName, testListGameName},
		})
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/aliases" {
					fmt.Fprintln(w, string(aliasesResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			aliases, err := d.GetAllAliasesByID(context.Background(), testGameIDString, nil)

			Convey("should return aliases", func() {
				So(err, ShouldBeNil)
				So(len(aliases), ShouldEqual, 2)
				So(aliases[0], ShouldEqual, testGameName)
				So(aliases[1], ShouldEqual, testListGameName)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/aliases" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.GetAllAliasesByID(context.Background(), testGameIDString, nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetAllAliasesByName(t *testing.T) {
	Convey("when calling GetAllAliasesByName", t, func() {
		aliasesResponse, _ := json.Marshal(Aliases{
			Aliases: []string{testGameName, testListGameName},
		})
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/aliases" {
					fmt.Fprintln(w, string(aliasesResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			aliases, err := d.GetAllAliasesByName(context.Background(), testGameName, nil)

			Convey("should return aliases", func() {
				So(err, ShouldBeNil)
				So(len(aliases), ShouldEqual, 2)
				So(aliases[0], ShouldEqual, testGameName)
				So(aliases[1], ShouldEqual, testListGameName)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/aliases" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.GetAllAliasesByName(context.Background(), testGameName, nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetLocalizations(t *testing.T) {
	localizationsResponse := readTestFile(localizationsResponseFile)

	Convey("when calling GetLocalizationsByID", t, func() {
		Convey("with no locale", func() {
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						fmt.Fprintln(w, string(localizationsResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				localizations, err := d.GetLocalizationsByID(context.Background(), testGameIDString, "", nil)

				Convey("should return a localization", func() {
					So(err, ShouldBeNil)
					So(localizations, ShouldResemble, testLocalizations)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetLocalizationsByID(context.Background(), testGameIDString, "", nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("with locale", func() {
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						fmt.Fprintln(w, string(localizationsResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				localizations, err := d.GetLocalizationsByID(context.Background(), testGameIDString, testLocale, nil)

				Convey("should return a localization", func() {
					So(err, ShouldBeNil)
					So(localizations, ShouldResemble, testLocalizations)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetLocalizationsByID(context.Background(), testGameIDString, testLocale, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})

	Convey("when calling GetLocalizationsByName", t, func() {
		Convey("with no locale", func() {
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						fmt.Fprintln(w, string(localizationsResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				localizations, err := d.GetLocalizationsByName(context.Background(), testGameName, "", nil)

				Convey("should return a localization", func() {
					So(err, ShouldBeNil)
					So(localizations, ShouldResemble, testLocalizations)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetLocalizationsByName(context.Background(), testGameName, "", nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
		Convey("with locale", func() {
			Convey("when successful", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						fmt.Fprintln(w, string(localizationsResponse))
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				localizations, err := d.GetLocalizationsByName(context.Background(), testGameName, testLocale, nil)

				Convey("should return a localization", func() {
					So(err, ShouldBeNil)
					So(localizations, ShouldResemble, testLocalizations)
				})
			})
			Convey("when server has an error", func() {
				ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
					if r.Method == "GET" && r.URL.Path == "/localizations" {
						w.WriteHeader(http.StatusInternalServerError)
					}
				}))
				defer ts.Close()

				d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
				if err != nil {
					t.Fatal(err)
				}

				_, err = d.GetLocalizationsByName(context.Background(), testGameName, testLocale, nil)

				Convey("should return an error", func() {
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestGetBulkLocalizationsByIDs(t *testing.T) {
	Convey("when calling GetBulkLocalizationsByIDs", t, func() {
		localizationsResponses := map[string]*Localizations{
			testGameIDString:     testLocalizations,
			testListGameIDString: testListLocalizations,
		}
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameID := r.URL.Query().Get("game_id")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameID])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByIDs(context.Background(), []string{testGameIDString, testListGameIDString}, "", nil)

			Convey("should return bulk localizations", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 2)
				So(bulkLocalizations[testGameIDString], ShouldResemble, testLocalizations)
				So(bulkLocalizations[testListGameIDString], ShouldResemble, testListLocalizations)
			})
		})
		Convey("handling duplicate slice entries", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameID := r.URL.Query().Get("game_id")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameID])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByIDs(context.Background(), []string{testGameIDString, testGameIDString, testListGameIDString}, "", nil)

			Convey("should return bulk localizations", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 2)
				So(bulkLocalizations[testGameIDString], ShouldResemble, testLocalizations)
				So(bulkLocalizations[testListGameIDString], ShouldResemble, testListLocalizations)
			})
		})
		Convey("handling empty slice", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameID := r.URL.Query().Get("game_id")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameID])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByIDs(context.Background(), []string{}, "", nil)

			Convey("should return empty slice", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 0)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.BulkGetLocalizationsByIDs(context.Background(), []string{testGameIDString, testListGameIDString}, "", nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetBulkLocalizationsByNames(t *testing.T) {
	Convey("when calling GetBulkLocalizationsByNames", t, func() {
		localizationsResponses := map[string]*Localizations{
			testGameName:     testLocalizations,
			testListGameName: testListLocalizations,
		}
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameName := r.URL.Query().Get("game_name")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameName])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByNames(context.Background(), []string{testGameName, testListGameName}, "", nil)

			Convey("should return bulk localizations", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 2)
				So(bulkLocalizations[testGameName], ShouldResemble, testLocalizations)
				So(bulkLocalizations[testListGameName], ShouldResemble, testListLocalizations)
			})
		})
		Convey("handling duplicate slice entries", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameName := r.URL.Query().Get("game_name")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameName])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByNames(context.Background(), []string{testGameName, testGameName, testListGameName}, "", nil)

			Convey("should return bulk localizations", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 2)
				So(bulkLocalizations[testGameName], ShouldResemble, testLocalizations)
				So(bulkLocalizations[testListGameName], ShouldResemble, testListLocalizations)
			})
		})
		Convey("handling empty slice", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					urlGameName := r.URL.Query().Get("game_name")
					localizationsResponse, _ := json.Marshal(localizationsResponses[urlGameName])
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			bulkLocalizations, err := d.BulkGetLocalizationsByNames(context.Background(), []string{}, "", nil)

			Convey("should return bulk localizations", func() {
				So(err, ShouldBeNil)
				So(len(bulkLocalizations), ShouldEqual, 0)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "GET" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.BulkGetLocalizationsByNames(context.Background(), []string{testGameName, testListGameName}, "", nil)

			Convey("should return error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestAddLocalization(t *testing.T) {
	localizationsResponse := readTestFile(localizationsResponseFile)

	Convey("when calling AddLocalizationByID", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "PUT" && r.URL.Path == "/localizations" {
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			localizations, err := d.AddLocalizationByID(context.Background(), testGameIDString, testLocale, testLocalizedName, nil)

			Convey("should return a localization", func() {
				So(err, ShouldBeNil)
				So(localizations, ShouldResemble, testLocalizations)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "PUT" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.AddLocalizationByID(context.Background(), testGameIDString, testLocale, testLocalizedName, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})

	Convey("when calling GetLocalizationsByName", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "PUT" && r.URL.Path == "/localizations" {
					fmt.Fprintln(w, string(localizationsResponse))
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			localizations, err := d.AddLocalizationByName(context.Background(), testGameName, testLocale, testLocalizedName, nil)

			Convey("should return a localization", func() {
				So(err, ShouldBeNil)
				So(localizations, ShouldResemble, testLocalizations)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "PUT" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = d.AddLocalizationByName(context.Background(), testGameName, testLocale, testLocalizedName, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestDeleteLocalization(t *testing.T) {
	Convey("when calling DeleteLocalizationByID", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusNoContent)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteLocalizationByID(context.Background(), testGameIDString, testLocale, nil)

			Convey("should return no error", func() {
				So(err, ShouldBeNil)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteLocalizationByID(context.Background(), testGameIDString, testLocale, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})

	Convey("when calling DeleteLocalizationByName", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusNoContent)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteLocalizationByName(context.Background(), testGameName, testLocale, nil)

			Convey("should return no error", func() {
				So(err, ShouldBeNil)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteLocalizationByName(context.Background(), testGameName, testLocale, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestDeleteAllLocalizations(t *testing.T) {
	Convey("when calling DeleteAllLocalizationsByID", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations/all" {
					w.WriteHeader(http.StatusNoContent)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteAllLocalizationsByID(context.Background(), testGameIDString, nil)

			Convey("should return no error", func() {
				So(err, ShouldBeNil)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations/all" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteAllLocalizationsByID(context.Background(), testGameIDString, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})

	Convey("when calling DeleteAllLocalizationsByName", t, func() {
		Convey("when successful", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations/all" {
					w.WriteHeader(http.StatusNoContent)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteAllLocalizationsByName(context.Background(), testGameName, nil)

			Convey("should return no error", func() {
				So(err, ShouldBeNil)
			})
		})
		Convey("when server has an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				if r.Method == "DELETE" && r.URL.Path == "/localizations/all" {
					w.WriteHeader(http.StatusInternalServerError)
				}
			}))
			defer ts.Close()

			d, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = d.DeleteAllLocalizationsByName(context.Background(), testGameName, nil)

			Convey("should return an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetUniques(t *testing.T) {
	Convey("when calling getUniques", t, func() {
		Convey("succeeds with empty slice", func() {
			values := getUniques([]string{})
			So(len(values), ShouldEqual, 0)
		})
		Convey("succeeds with one element", func() {
			testValues := []string{"1"}
			values := getUniques(testValues)
			So(len(values), ShouldEqual, 1)
			So(values, ShouldResemble, testValues)
		})
		Convey("succeeds with no duplicates", func() {
			testValues := []string{"1", "2", "3"}
			values := getUniques(testValues)
			So(len(values), ShouldEqual, 3)
			So(values, ShouldResemble, testValues)
		})
		Convey("succeeds with duplicates", func() {
			testValues := []string{"1", "1", "2", "1", "3"}
			testValuesNoDupes := []string{"1", "2", "3"}
			values := getUniques(testValues)
			So(len(values), ShouldEqual, 3)
			So(values, ShouldResemble, testValuesNoDupes)
		})
	})
}
