package main

import (
	"code.justin.tv/chat/db"
	"code.justin.tv/common/config"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/common/golibs/errorlogger/rollbar"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/streams"
	"errors"
	"fmt"
	"github.com/cactus/go-statsd-client/statsd"
	_ "github.com/lib/pq"
	"github.com/zenazn/goji/graceful"
	"golang.org/x/net/context"
	"log"
	"os"
	"strings"
	"time"
)

var (
	rollbarToken = "f49d1714480b469694415113c32c0fba"
)

const (
	versionsKept = 30
)

var errLogger errorlogger.ErrorLogger
var stats statsd.Statter

func init() {
	config.Register(map[string]string{
		"jax-base-url": "http://jax-internal-staging.us-west2.justin.tv",
	})
}

func main() {
	config.Parse()

	// Remove timestamps from log entries
	log.SetFlags(0)
	log.Printf("Starting discovery_jaxstore with environment: %s", config.Environment())

	graceful.HandleSignals()
	graceful.PreHook(func() {
		log.Printf("Shutdown signal was received.")
	})

	go func() {
		errLogger = rollbar.NewRollbarLogger(rollbarToken, config.Environment())
		stats = configureStatsd()

		jax := streams.NewClient(config.MustResolve("jax-base-url"), stats)
		masterOpts := backend.CronDBOptions()

		master, err := db.Open(masterOpts...)
		if err != nil {
			log.Fatalf("Error opening master db: %s", err.Error())
		}
		log.Printf("Database connection was established")

		master.SetCallbacks(func(evName, queryName string, d time.Duration) {
			bucket := fmt.Sprintf("%s.%s", evName, queryName)
			stats.Inc(bucket, 1, 0.1)
			stats.TimingDuration(bucket, d, 0.1)
		}, func(evName string) {
			stats.Inc(evName, 1, 0.1)
		})

		ticker := time.NewTicker(15 * time.Second)
		for {
			<-ticker.C
			storeJaxGames(jax, master)
			dropOldVersions(master)
		}
		log.Printf("Initiating manual shutdown after for loop was broken.")
		graceful.Shutdown()
	}()
	graceful.Wait()
}

func storeJaxGames(streamsClient streams.Client, db db.DB) {
	jaxGames, err := streamsClient.TopGames(context.Background(), "")
	if err != nil {
		errLogger.Error(err)
		return
	}

	if len(jaxGames) == 0 {
		errorMessage := "No top games from JAX. Expecting at least 1 game."
		log.Printf("NOTICE: %s", errorMessage)
		errLogger.Error(errors.New(errorMessage))
		return
	}

	query := `SELECT nextval('top_games_version'); INSERT INTO top_games (game_id, viewers, channels, version) VALUES `
	top_games := []string{}
	for _, g := range jaxGames {
		top_games = append(top_games, fmt.Sprintf("(%d, %d, %d, currval('top_games_version'))", g.ID, g.Viewers, g.Channels))
	}
	query = query + strings.Join(top_games, ",")
	_, err = db.Exec(context.Background(), "store_top_games", query)
	if err != nil {
		errLogger.Error(err)
		stats.Inc("top_games_update.error", 1, 1)
	} else {
		stats.Inc("top_games_update.success", int64(len(top_games)), 1)
	}
}

func dropOldVersions(db db.DB) {
	rows, err := db.Query(context.Background(), "recent_versions", "SELECT DISTINCT(version) FROM top_games ORDER BY version DESC")
	if err != nil {
		log.Printf("error getting recent versions: %s", err.Error())
		errLogger.Error(err)
		return
	}
	versions := []int64{}
	var version int64
	for rows.Next() {
		err := rows.Scan(&version)
		if err == nil {
			versions = append(versions, version)
		} else {
			log.Printf("error reading value %s", err.Error())
		}
	}
	rows.Close()

	if len(versions) > versionsKept {
		versionList := "("
		for _, v := range versions[versionsKept:] {
			versionList = fmt.Sprintf("%s%d,", versionList, v)
		}
		versionList = versionList[:len(versionList)-1] + ")"

		_, err := db.Exec(context.Background(), "delete_old_versions", "DELETE FROM top_games WHERE version IN "+versionList)
		if err != nil {
			log.Printf("error deleting old versions: %s", err.Error())
			errLogger.Error(err)
		}
	}
}

func configureStatsd() statsd.Statter {
	var stats statsd.Statter

	statsdHostport := os.Getenv("STATSD_HOSTPORT")
	if len(statsdHostport) > 0 {
		hostName, err := os.Hostname()
		if err != nil {
			hostName = "unknown_host"
		}

		prefix := fmt.Sprintf("discovery.%v.%v", config.Environment(), hostName)

		if stats, err = statsd.NewBufferedClient(statsdHostport, prefix, 1*time.Second, 512); err != nil {
			log.Fatalf("StatsD configuration error: %v", err)
		}

		log.Printf("Connected to StatsD at %s\n", statsdHostport)
	} else {
		stats, _ = statsd.NewNoopClient()
	}

	return stats
}
