// Game objects.
//
// Game contains all fields from the games database,
// as well as art links.
// LiveGame augments a Game with live channel and viewers information.

package game

import (
	"fmt"
	"net/url"
)

const (
	artBase     = "https://static-cdn.jtvnw.net/"
	nameFormat  = `%s-%vx%v.jpg`
	boxArtBase  = artBase + "ttv-boxart/" + nameFormat
	logoArtBase = artBase + "ttv-logoart/" + nameFormat
)

type LiveGame struct {
	Channels int  `json:"channels"`
	Viewers  int  `json:"viewers"`
	Game     Game `json:"game"`
}

type Game struct {
	Name        string            `json:"name"`
	ID          int               `json:"id"`
	GiantbombID int               `json:"giantbombId"`
	Images      map[string]string `json:"images"`
	Popularity  int               `json:"popularity"`
	Properties  map[string]string `json:"properties"`
	BoxArt      map[string]string `json:"boxArt"`
	LogoArt     map[string]string `json:"logoArt"`
}

type LocalizationData struct {
	GameID int    `json:"game_id"`
	Name   string `json:"localized_name"`
	Locale string `json:"locale"`
}

type Localizations struct {
	GameID         int               `json:"game_id"`
	LocalizedNames map[string]string `json:"localizations"`
}

var (
	boxSizes = map[string]map[string]int{
		"small":  map[string]int{"width": 52, "height": 72},
		"medium": map[string]int{"width": 136, "height": 190},
		"large":  map[string]int{"width": 272, "height": 380},
	}
	logoSizes = map[string]map[string]int{
		"small":  map[string]int{"width": 60, "height": 36},
		"medium": map[string]int{"width": 120, "height": 72},
		"large":  map[string]int{"width": 240, "height": 144},
	}
	template = map[string]string{"width": "{width}", "height": "{height}"}
)

// NewGame returns a game object with a placeholder name
// and intialized empty maps.
func NewGame() Game {
	g := Game{}
	g.Name = "<NAME OF GAME>"
	g.Images = make(map[string]string)
	g.Properties = make(map[string]string)

	return g
}

// buildArt populates the BoxArt and LogoArt fields of the game.
// It might be nice to export this stuff to external config.
func (g *Game) BuildArt() {
	// This is so dumb: url.Values.Encode encodes spaces as "+",
	// which doesn't work for our images.
	// It encodes ":" differently from old discovery but the link works so whatever.
	u := &url.URL{Path: g.Name}
	urlName := u.String()

	g.BoxArt = make(map[string]string)
	g.LogoArt = make(map[string]string)
	for k, v := range boxSizes {
		g.BoxArt[k] = fmt.Sprintf(boxArtBase, urlName, v["width"], v["height"])
	}
	for k, v := range logoSizes {
		g.LogoArt[k] = fmt.Sprintf(logoArtBase, urlName, v["width"], v["height"])
	}
	g.BoxArt["template"] = fmt.Sprintf(boxArtBase, urlName, template["width"], template["height"])
	g.LogoArt["template"] = fmt.Sprintf(logoArtBase, urlName, template["width"], template["height"])
}
