package giantbomb

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/game"
	"github.com/cactus/go-statsd-client/statsd"

	"github.com/afex/hystrix-go/hystrix"

	"golang.org/x/net/context"
)

const (
	// Giantbomb sometimes takes a long time...
	reqTimeout = 10 * time.Second
)

// Client is an interface for querying Giantbomb.com.
type Client interface {
	FetchGame(ctx context.Context, id int) (GBGame, error)
	FetchRecentGames(ctx context.Context, offset, limit int) ([]GBGame, error)
}

type gb struct {
	baseURL string
	apiKey  string

	stats statsd.Statter
}

var _ Client = new(gb)

const (
	fetchGame        = "fetch_game"
	fetchRecentGames = "fetch_recent_games"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		fetchGame: hystrix.CommandConfig{
			Timeout:               10000,
			ErrorPercentThreshold: 25,
		},
		fetchRecentGames: hystrix.CommandConfig{
			Timeout:               10000,
			ErrorPercentThreshold: 25,
		},
	})
}

// GBGame is the game structure returned from giantbomb.
type GBGame struct {
	game.Game
	Aliases      []string  `json:"aliases"`
	Genres       []string  `json:"genres"`
	Themes       []string  `json:"themes"`
	Franchises   []string  `json:"franchises"`
	SimilarGames []string  `json:"similar_games"`
	LastUpdated  time.Time `json:"date_last_updated"`
}

// NewClient creates a giantbomb client using the default implementation.
func NewClient(baseURL, apiKey string, stats statsd.Statter) Client {
	client := &gb{
		baseURL: baseURL,
		apiKey:  apiKey,
		stats:   stats,
	}

	return client
}

func toStringMap(in map[string]interface{}) map[string]string {
	out := make(map[string]string)

	for k, v := range in {
		value, ok := v.(string)
		if ok {
			out[k] = value
		}
	}

	return out
}

func stringKeysFrom(in []interface{}, key string) []string {
	array := []string{}
	for _, m := range in {
		mm := m.(map[string]interface{})
		if v, ok := mm[key]; ok {
			array = append(array, v.(string))
		}
	}
	return array
}

func readAliases(aliases string) []string {
	csvstring := strings.Replace(aliases, "\n", ",", -1)
	split := strings.Split(csvstring, ",")
	for i := len(split) - 1; i >= 0; i-- {
		if split[i] == "" {
			copy(split[i:], split[i+1:])
			split = split[:len(split)-1]
		}
	}
	return split
}

// ErrGameNotFound represents a 404 from giantbomb
var ErrGameNotFound = fmt.Errorf("Game not found on giantbomb")

// ErrMalformedResponse will be thrown if the giantbomb response format changes.
var ErrMalformedResponse = fmt.Errorf("Giantbomb response not in expected format")

func (T *gb) baseParams() url.Values {
	return url.Values{
		"api_key": []string{T.apiKey},
		"format":  []string{"json"},
	}
}

func (T *gb) giantbombRequest(ctx context.Context, path string, opts url.Values) (*http.Response, error) {
	req, err := http.NewRequest("GET", T.baseURL, nil)
	if err != nil {
		return nil, err
	}

	req.URL.Path = path
	req.URL.RawQuery = opts.Encode()

	client := chitin.Client(ctx)
	client.Timeout = reqTimeout
	return client.Do(req)
}

func (T *gb) fetchResults(ctx context.Context, path string, opts url.Values) (map[string]interface{}, error) {
	var results map[string]interface{}

	resp, err := T.giantbombRequest(ctx, path, opts)
	if err != nil {
		return results, err
	}
	defer resp.Body.Close()

	decoder := json.NewDecoder(resp.Body)
	err = decoder.Decode(&results)

	return results, nil
}

func (T *gb) FetchGame(ctx context.Context, id int) (GBGame, error) {
	g := GBGame{}

	output := make(chan map[string]interface{})
	errors := hystrix.Go(fetchGame, func() error {
		results, err := T.fetchResults(ctx, "/api/game/"+strconv.Itoa(id)+"/", T.baseParams())
		if err != nil {
			return err
		}
		output <- results
		return nil
	}, nil)

	select {
	case results := <-output:
		numResults, ok := results["number_of_total_results"].(float64)
		if numResults == 0 {
			return g, ErrGameNotFound
		} else if !ok {
			return g, ErrMalformedResponse
		}

		results = results["results"].(map[string]interface{})
		return newGBGame(results), nil
	case err := <-errors:
		return g, err
	}
}

func (T *gb) FetchRecentGames(ctx context.Context, offset, limit int) ([]GBGame, error) {
	g := []GBGame{}

	output := make(chan map[string]interface{})
	errors := hystrix.Go(fetchRecentGames, func() error {
		opts := T.baseParams()
		opts.Add("offset", strconv.Itoa(offset))
		opts.Add("limit", strconv.Itoa(limit))
		opts.Add("sort", `date_last_updated:desc`)

		results, err := T.fetchResults(ctx, "/api/games/", opts)
		if err != nil {
			return err
		}
		output <- results
		return nil
	}, nil)

	select {
	case results := <-output:
		numResults, ok := results["number_of_page_results"].(float64)
		if numResults == 0 {
			return g, ErrGameNotFound
		} else if !ok {
			return g, ErrMalformedResponse
		}

		resultArray := results["results"].([]interface{})

		for i := 0; i < int(numResults); i++ {
			result := resultArray[i].(map[string]interface{})
			g = append(g, newGBGame(result))
		}

		return g, nil
	case err := <-errors:
		return g, err
	}
}

func newGBGame(result map[string]interface{}) GBGame {
	g := GBGame{
		Game: game.NewGame(),
	}

	g.Game.Name = result["name"].(string)
	g.Game.GiantbombID = int(result["id"].(float64))

	g.LastUpdated, _ = time.Parse("2006-01-02 15:04:05", result["date_last_updated"].(string))

	if result["image"] != nil {
		g.Game.Images = toStringMap(result["image"].(map[string]interface{}))
	}
	if result["aliases"] != nil {
		g.Aliases = readAliases(result["aliases"].(string))
	}
	if result["franchises"] != nil {
		g.Franchises = stringKeysFrom(result["franchises"].([]interface{}), "name")
	}
	if result["themes"] != nil {
		g.Themes = stringKeysFrom(result["themes"].([]interface{}), "name")
	}
	if result["genres"] != nil {
		g.Genres = stringKeysFrom(result["genres"].([]interface{}), "name")
	}
	if result["similar_games"] != nil {
		g.SimilarGames = stringKeysFrom(result["similar_games"].([]interface{}), "name")
	}

	return g
}
