package main

import (
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"strconv"
	"time"

	"code.justin.tv/common/config"
	"code.justin.tv/web/discovery/aws"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/giantbomb"
	"code.justin.tv/web/discovery/s3"
	"code.justin.tv/web/discovery/streams"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/zenazn/goji/graceful"
	"golang.org/x/net/context"
)

var (
	// loggers
	errorLogger = log.New(os.Stderr, "", log.LstdFlags)
	infoLogger  = log.New(os.Stdout, "", log.LstdFlags)

	// S3 location
	gameDumpBucket  = "twitch-gds-gamesdb-dump"
	gameDumpFile    = "games.csv"
	aliasesDumpFile = "aliases.csv"
)

const (
	// Giantbomb only returns up to 100 results, so this should be <= 100
	indexChunkSize = 100
)

func init() {
	config.Register(map[string]string{
		"port": "9292",
		"time": "168", // 1 week

		"jax-base-url":      "http://jax-internal-elb.prod.us-west2.justin.tv",
		"giantbomb-url":     "http://www.giantbomb.com",
		"giantbomb-api-key": "",
	})
}

func main() {
	config.Parse()
	log.SetOutput(os.Stdout)

	graceful.HandleSignals()
	graceful.PreHook(func() {
		infoLogger.Printf("Shutdown signal was received.")
	})

	go func() {
		// for non-production, change name of bucket because s3 bucket names have to be globally unique
		if config.Environment() != "production" {
			gameDumpBucket += fmt.Sprintf("-%s", config.Environment())
		}

		stats := config.Statsd()
		jax := streams.NewClient(config.MustResolve("jax-base-url"), stats)
		gbURL := config.MustResolve("giantbomb-url")
		gbKey := config.MustResolve("giantbomb-api-key")
		updatePeriod, err := strconv.Atoi(config.MustResolve("time"))
		if err != nil {
			errorLogger.Fatalf("TIME should be an integer")
		}

		b, err := backend.NewBackend(
			jax,
			stats,
		)
		if err != nil {
			errorLogger.Fatalf("Error creating backend: %v\n", err)
		}

		gd := s3.NewGameDumper(stats, aws.CreateAwsCredentials())

		// index all initial run
		indexAll(giantbomb.NewClient(gbURL, gbKey, stats), "http://localhost:"+config.MustResolve("port"), updatePeriod, stats)
		infoLogger.Printf("Done indexing all")

		// dump games initial run
		err = dumpGamesToS3(b, gd, stats)
		if err != nil {
			errorLogger.Fatalf("Error dumping games into S3: %s", err.Error())
		}
		infoLogger.Printf("Done dumping games into S3")

		// index popularity initial run
		err = indexPopularity(b, jax, stats)
		if err != nil {
			errorLogger.Fatalf("Error indexing popularity: %s", err.Error())
		}
		infoLogger.Printf("Done indexing popularity")

		// index all / popularity in intervals
		allTicker := time.NewTicker(24 * time.Hour)
		popularityTicker := time.NewTicker(5 * time.Minute)

		for {
			select {
			case <-allTicker.C:
				indexAll(giantbomb.NewClient(gbURL, gbKey, stats), "http://localhost:"+config.MustResolve("port"), updatePeriod, stats)
				infoLogger.Printf("Done indexing all")

				err = dumpGamesToS3(b, gd, stats)
				if err != nil {
					errorLogger.Fatalf("Error dumping games into S3: %s", err.Error())
				}
				infoLogger.Printf("Done dumping games into S3")
			case <-popularityTicker.C:
				err = indexPopularity(b, jax, stats)
				if err != nil {
					errorLogger.Fatalf("Error indexing popularity: %s", err.Error())
				}
				infoLogger.Printf("Done indexing popularity")
			}
		}

		infoLogger.Printf("Initiating manual shutdown after for loop was broken.")
		graceful.Shutdown()
	}()

	graceful.Wait()
}

// dumpGamesToS3 gets the id, name, and giantbomb_id for all games
// and dumps it into S3 in csv format. Also dumps all game aliases to S3 in csv format.
func dumpGamesToS3(b *backend.Backend, gd s3.GameDumper, stats statsd.Statter) error {
	mappings, err := b.GameIDMappingCSV(context.Background())
	if err != nil {
		stats.Inc("games_dump.games.db_error", 1, 1)
		infoLogger.Printf(err.Error())
	} else {
		err = gd.DumpToS3(mappings, gameDumpBucket, gameDumpFile)
		stats.Inc("games_dump.games.dump", 1, 1)
		if err != nil {
			stats.Inc("games_dump.games.s3_error", 1, 1)
			infoLogger.Printf(err.Error())
		}
	}

	aliases, err := b.AllAliasesCSV(context.Background())
	if err != nil {
		stats.Inc("games_dump.aliases.db_error", 1, 1)
		infoLogger.Printf(err.Error())
	} else {
		err = gd.DumpToS3(aliases, gameDumpBucket, aliasesDumpFile)
		stats.Inc("games_dump.aliases.dump", 1, 1)
		if err != nil {
			stats.Inc("games_dump.aliases.s3_error", 1, 1)
			infoLogger.Printf(err.Error())
		}
	}

	return nil
}

// indexPopularity gets popularities of games from giantbomb,
// and updates games on the database.
func indexPopularity(b *backend.Backend, streamsClient streams.Client, stats statsd.Statter) error {
	timingStart := time.Now()
	liveGames, err := streamsClient.LiveGames(context.Background())
	stats.TimingDuration("popularity.retrieve_live_games", time.Since(timingStart), 1)

	if err != nil {
		stats.Inc("popularity.update_error", 1, 1)
		return err
	}

	ids, err := b.GetPopularGameIDs(context.Background())
	if err != nil {
		return err
	}

	// Set games not being played right now to 0 viewers
	for _, id := range ids {
		if _, ok := liveGames[id]; !ok {
			liveGames[id] = streams.JaxGame{
				ID:      id,
				Viewers: 0,
			}
		}
	}

	for id, g := range liveGames {
		err := b.UpdateGamePopularity(context.Background(), id, g.Viewers)
		if err != nil {
			errorLogger.Printf("Failed to update popularity for id %v: %s", id, err.Error())
		} else {
			infoLogger.Printf("Updated popularity for id %v, with %v viewers", id, g.Viewers)
		}
	}

	numLiveGamesUpdated := len(liveGames)
	stats.Inc("popularity.update_success", int64(numLiveGamesUpdated), 1)
	infoLogger.Printf("Updated %v popularity values", numLiveGamesUpdated)
	return nil
}

// indexAll goes fetches games from giantbomb that have been updated recently,
// and adds/updates them via the discovery service.
func indexAll(gbClient giantbomb.Client, discoveryURL string, updatePeriod int, stats statsd.Statter) {
	timeLimit := time.Now().Add(time.Duration(-updatePeriod) * time.Hour)
	offset := 0
	for {
		timingStart := time.Now()
		games, err := gbClient.FetchRecentGames(context.Background(), offset, indexChunkSize)
		stats.TimingDuration("all.giantbomb.fetch_recent_games", time.Since(timingStart), 1)

		offset += indexChunkSize
		if err != nil {
			stats.Inc("all.giantbomb.fetch_recent_games_error", 1, 0.1)
			if err == giantbomb.ErrGameNotFound {
				return
			}
			errorLogger.Fatalf("failed to fetch games from giantbomb for offset %v %s\n", offset, err.Error())
			continue
		}

		stats.Inc("all.giantbomb.fetch_recent_games_success", int64(len(games)), 1)

		for _, g := range games {
			if g.LastUpdated.Before(timeLimit) {
				return
			}

			req, _ := http.NewRequest("PUT", discoveryURL+"/games?giantbomb_id="+strconv.Itoa(g.GiantbombID), nil)
			resp, err := http.DefaultClient.Do(req)

			if err != nil {
				stats.Inc("all.giantbomb.unknown_error", 1, 1)
				errorLogger.Printf("Warning: failed to update %v: %s", g.GiantbombID, err.Error())
			} else {
				defer resp.Body.Close()
				if resp.StatusCode != 200 {
					stats.Inc(fmt.Sprintf("all.giantbomb.status_code_error.%d", resp.StatusCode), 1, 1)
					b, _ := ioutil.ReadAll(resp.Body)
					errorLogger.Printf("Warning: failed to update %v: %v %s\n", g.GiantbombID, resp.StatusCode, string(b))
				} else {
					stats.Inc("all.giantbomb.update_success", 1, 1)
					infoLogger.Printf("updated %v %s", g.GiantbombID, g.Name)
				}
			}
		}

		// Reach the end of the giantbomb list, this should probably never happen
		if len(games) < indexChunkSize {
			stats.Inc("all.giantbomb.end_of_list_error", 1, 1)
			return
		}
	}
}
