package kinesis

import (
	"encoding/json"
	"log"
	"strconv"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsKinesis "github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/cactus/go-statsd-client/statsd"
)

const GameIndex = "game"
const KinesisErrStatName = "kinesis.errors"

type Record struct {
	Action   string            `json:"action"`
	Document Document          `json:"document"`
	Options  map[string]string `json:"options"`
}

type Document struct {
	ID     string  `json:"id"`
	Type   string  `json:"type"`
	Fields []Field `json:"fields"`
}

type Field struct {
	Name  string      `json:"name"`
	Value interface{} `json:"value"`
	Type  string      `json:"type"`
}

type GameIndexer interface {
	AddGame(ID, popularity int, name string, aliases []string, localizations map[string]string) error
	DeleteGame(ID int) error
}

type kinesis struct {
	client     *awsKinesis.Kinesis
	statsd     statsd.Statter
	streamName string
}

func NewGameIndexer(streamName string, statsd statsd.Statter, creds *credentials.Credentials) GameIndexer {
	sess := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	client := awsKinesis.New(sess)

	k := &kinesis{
		client:     client,
		statsd:     statsd,
		streamName: streamName,
	}
	return k
}

func (K *kinesis) putRecord(ID int, rec Record) error {
	payload, err := json.Marshal(rec)
	if err != nil {
		return err
	}

	pKey := strconv.Itoa(ID)
	params := &awsKinesis.PutRecordInput{
		Data:         payload,
		PartitionKey: aws.String(pKey),
		StreamName:   aws.String(K.streamName),
	}
	_, err = K.client.PutRecord(params)

	if err != nil {
		K.statsd.Inc(KinesisErrStatName, 1, 1)
		log.Printf("Kinesis error (%s) for payload: %s", err.Error(), string(payload))
		return err
	}

	return nil
}

func (K *kinesis) AddGame(ID, popularity int, name string, aliases []string, localizations map[string]string) error {
	fields := []Field{
		Field{Name: "name", Value: name, Type: "string"},
		Field{Name: "popularity", Value: popularity, Type: "integer"},
		Field{Name: "aliases", Value: aliases, Type: "list"},
		Field{Name: "localizations", Value: localizations, Type: "map"},
	}

	doc := Document{
		ID:     strconv.Itoa(ID),
		Type:   GameIndex,
		Fields: fields,
	}

	rec := Record{
		Action:   "update",
		Document: doc,
	}

	return K.putRecord(ID, rec)
}

func (K *kinesis) DeleteGame(ID int) error {
	doc := Document{
		ID:     strconv.Itoa(ID),
		Type:   GameIndex,
		Fields: []Field{},
	}

	rec := Record{
		Action:   "delete",
		Document: doc,
	}

	return K.putRecord(ID, rec)
}
