package main

import (
	"fmt"
	"log"
	"net/http"
	"os"
	"time"

	"golang.org/x/net/context"

	_ "expvar"
	_ "net/http/pprof"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/getsentry/raven-go"

	"code.justin.tv/common/config"
	"code.justin.tv/common/golibs/errorlogger/rollbar"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/web/discovery/api"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/giantbomb"
	"code.justin.tv/web/discovery/streams"
	"code.justin.tv/web/service_template/handler"
)

var (
	rollbarToken = "f49d1714480b469694415113c32c0fba" // FIXME: move into environment variable
)

func init() {
	config.Register(map[string]string{
		"addr":              ":8080",
		"hystrix-addr":      ":7001",
		"raven-dsn":         "",
		"jax-base-url":      "http://jax-internal-production.us-west2.justin.tv",
		"giantbomb-url":     "http://www.giantbomb.com",
		"giantbomb-api-key": "",
	})
}

func main() {
	config.Parse()

	// Remove timestamps from log entries
	log.SetFlags(0)

	stats := configureStatsd()
	hystrixHandler := configureHystrix()

	b := makeBackend(stats)
	apiMux, err := api.NewAPI(
		b,
		stats,
		configureRaven(),
		giantbomb.NewClient(config.MustResolve("giantbomb-url"), config.MustResolve("giantbomb-api-key"), stats),
		rollbar.NewRollbarLogger(rollbarToken, config.Environment()), // FIXME: use config's ErrorLogger
	)
	if err != nil {
		log.Fatal(err)
	}
	http.Handle("/", apiMux)

	go http.ListenAndServe(config.MustResolve("hystrix-addr"), hystrixHandler)

	gometrics.Monitor(stats, time.Second*5)

	handler.BeginServer(http.DefaultServeMux, config.MustResolve("addr"))
}

func configureHystrix() *hystrix.StreamHandler {
	handler := hystrix.NewStreamHandler()
	handler.Start()

	return handler
}

func configureStatsd() statsd.Statter { // FIXME: switch to use config.Statsd()
	var stats statsd.Statter

	statsdHostport := os.Getenv("STATSD_HOSTPORT")
	if len(statsdHostport) > 0 {
		hostName, err := os.Hostname()
		if err != nil {
			hostName = "unknown_host"
		}

		prefix := fmt.Sprintf("discovery.%v.%v", config.Environment(), hostName)

		if stats, err = statsd.NewBufferedClient(statsdHostport, prefix, 1*time.Second, 512); err != nil {
			log.Fatalf("StatsD configuration error: %v", err)
		}

		log.Printf("Connected to StatsD at %s\n", statsdHostport)

		c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
			StatsdAddr: statsdHostport,
			Prefix:     fmt.Sprintf("discovery.%s.hystrix.%s", config.Environment(), hostName),
		})
		if err != nil {
			log.Fatalf("could not initialize statsd client for hystrix: %v", err)
		}

		metricCollector.Registry.Register(c.NewStatsdCollector)
	} else {
		stats, _ = statsd.NewNoopClient()
	}

	return stats
}

func configureRaven() *raven.Client {
	var ravenClient *raven.Client
	var err error
	if dsn := config.Resolve("raven-dsn"); len(dsn) > 0 {
		ravenClient, err = raven.NewClient(dsn, nil)
		if err != nil {
			log.Fatalf("Error creating raven client: %s\n", err.Error())
		} else {
			log.Printf("Created raven client\n")
		}
	} else {
		log.Println("Not logging to Sentry (Missing RAVEN_DSN var)")
	}
	return ravenClient
}

func makeBackend(stats statsd.Statter) *backend.Backend {
	jax := streams.NewClient(config.MustResolve("jax-base-url"), stats)

	b, err := backend.NewBackend(
		jax,
		stats,
	)

	if err != nil {
		log.Fatalf("Error creating backend: %v\n", err)
	}

	go func() {
		for {
			time.Sleep(10 * time.Second)
			b.Health(context.Background())
		}
	}()

	return b
}
