package s3

import (
	"log"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsS3 "github.com/aws/aws-sdk-go/service/s3"
	"github.com/cactus/go-statsd-client/statsd"
)

const s3SuccStatName = "s3.success"
const s3ErrStatName = "s3.errors"

// GameDumper is an interface for dumping game data
type GameDumper interface {
	DumpToS3(body, bucket, key string) error
}

type s3 struct {
	client *awsS3.S3
	statsd statsd.Statter
}

// NewGameDumper instantiates and returns a new GameDumper
func NewGameDumper(statsd statsd.Statter, creds *credentials.Credentials) GameDumper {
	sess := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	client := awsS3.New(sess)

	s := &s3{
		client: client,
		statsd: statsd,
	}
	return s
}

func (S3 *s3) DumpToS3(body, bucket, key string) error {
	err := S3.createBucket(bucket)
	if err != nil {
		S3.statsd.Inc(s3ErrStatName, 1, 1)
		return err
	}

	err = S3.putObject(body, bucket, key)
	if err != nil {
		S3.statsd.Inc(s3ErrStatName, 1, 1)
		return err
	}

	S3.statsd.Inc(s3SuccStatName, 1, 1)
	return nil
}

func (S3 *s3) createBucket(bucket string) error {
	bucketOutput, err := S3.client.ListBuckets(&awsS3.ListBucketsInput{})
	if err != nil {
		log.Println("Failed to get buckets", err)
		return err
	}

	for _, b := range bucketOutput.Buckets {
		if *b.Name == bucket {
			log.Println("Bucket already exists")
			return nil
		}
	}

	_, err = S3.client.CreateBucket(&awsS3.CreateBucketInput{
		Bucket: &bucket,
	})
	if err != nil {
		log.Println("Failed to create bucket", err)
		return err
	}

	return nil

}

func (S3 *s3) putObject(body, bucket, key string) error {
	_, err := S3.client.PutObject(&awsS3.PutObjectInput{
		Body:   strings.NewReader(body),
		Bucket: &bucket,
		Key:    &key,
	})
	if err != nil {
		log.Printf("Failed to upload data to %s/%s, %s\n", bucket, key, err)
		return err
	}

	return nil
}
