package sns

import (
	"log"
	"strconv"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	awsSNS "github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"
)

const SNSErrStatName = "snsErrors"

type GameCommunity interface {
	UpdateGame(id int, name string, action string) error
}

type sns struct {
	client *awsSNS.SNS
	statsd statsd.Statter
	topic  string
}

func NewGameCommunity(topic string, statsd statsd.Statter, creds *credentials.Credentials) GameCommunity {
	sess := session.New(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	client := awsSNS.New(sess)

	return &sns{
		client: client,
		statsd: statsd,
		topic:  topic,
	}
}

func (s *sns) UpdateGame(id int, name string, action string) error {
	params := &awsSNS.PublishInput{
		Message:  aws.String(strconv.Itoa(id)),
		TopicArn: aws.String(s.topic),
		MessageAttributes: map[string]*awsSNS.MessageAttributeValue{
			"action": {
				StringValue: aws.String(action),
				DataType:    aws.String("String"),
			},
			"name": {
				StringValue: aws.String(name),
				DataType:    aws.String("String"),
			},
		},
	}

	_, err := s.client.Publish(params)

	if err != nil {
		s.statsd.Inc(SNSErrStatName, 1, 1)
		log.Printf("SNS error (%s) for action: %s and id: %s", err.Error(), action, strconv.Itoa(id))
		return err
	}

	return nil
}
