// Takes care of talking to jax.

package streams

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/common/chitin"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

const (
	reqTimeout            = 1 * time.Second
	streamSummaryEndPoint = "/stream/summary"
	cacheTime             = 15 * time.Second
)

// Client is an interface for querying Jax.
type Client interface {
	LiveGames(ctx context.Context) (map[int]JaxGame, error)
	LiveGameInfo(ctx context.Context, ids string) (map[int]JaxGame, error)
	TopGames(ctx context.Context, ordering string) ([]JaxGame, error)
}

type jax struct {
	baseURL        string
	stats          statsd.Statter
	cachedTopGames []JaxGame
}

var _ Client = new(jax)

// ErrJaxRequestFailed represents a failed request to jax.
type ErrJaxRequestFailed struct {
	Message string
}

func (e ErrJaxRequestFailed) Error() string {
	return fmt.Sprintf("Jax request failed: %s", e.Message)
}

// NewClient creates a jax client using the default implementation.
func NewClient(baseURL string, stats statsd.Statter) Client {
	client := &jax{
		baseURL:        baseURL,
		stats:          stats,
		cachedTopGames: []JaxGame{},
	}

	return client
}

const (
	cmdLiveInfo  = "jax_live_game_info"
	cmdLiveGames = "jax_live_games"
	cmdTopGames  = "jax_top_games"

	cmdLiveInfoCacheKey  = "jax_live_game_info"
	cmdLiveGamesCacheKey = "jax_live_games"
	cmdTopGamesCacheKey  = "jax_top_games"

	cmdLiveInfoFallbackCacheKey  = "jax_live_game_info_fallback"
	cmdLiveGamesFallbackCacheKey = "jax_live_games_fallback"
	cmdTopGamesFallbackCacheKey  = "jax_top_games_fallback"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		cmdLiveInfo: hystrix.CommandConfig{
			Timeout:               300,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 50,
		},
		cmdLiveInfoFallbackCacheKey: hystrix.CommandConfig{
			Timeout:               100,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 50,
		},
		cmdLiveGames: hystrix.CommandConfig{
			Timeout:               300,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 100,
		},
		cmdLiveGamesFallbackCacheKey: hystrix.CommandConfig{
			Timeout:               100,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 100,
		},
		cmdTopGames: hystrix.CommandConfig{
			Timeout:               300,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 100,
		},
		cmdTopGamesFallbackCacheKey: hystrix.CommandConfig{
			Timeout:               50,
			ErrorPercentThreshold: 25,
			MaxConcurrentRequests: 100,
		},
	})
}

// performRequest makes a GET request to jax, with a particular endpoint and url params.
// Simply returns the raw response.
func (T *jax) performRequest(ctx context.Context, endpoint string, opts url.Values) (*http.Response, error) {
	req, err := http.NewRequest("GET", T.baseURL, nil)
	if err != nil {
		return nil, err
	}

	req.URL.Path = endpoint
	req.URL.RawQuery = opts.Encode()

	client := chitin.Client(ctx)
	client.Timeout = reqTimeout

	t := time.Now()

	resp, err := client.Do(req)
	if err != nil {
		// Don't record timing stats for request errors,
		// since they're likely timeouts which won't give us useful timings.
		T.stats.Inc(fmt.Sprintf("streams.%s.error", endpoint), 1, 0.1)
		return resp, err
	}

	T.stats.TimingDuration(fmt.Sprintf("streams.%s", endpoint), time.Since(t), 0.1)
	T.stats.Inc(fmt.Sprintf("streams.%s.%d", endpoint, resp.StatusCode), 1, 0.1)
	if resp.StatusCode != http.StatusOK {
		return resp, ErrJaxRequestFailed{Message: resp.Status}
	}

	return resp, nil
}

type summary struct {
	Games []JaxGame `json:"results"`
}

// JaxGame contains live info for a game id.
type JaxGame struct {
	Channels int `json:"channels"`
	ID       int `json:"rails.game_id"`
	Viewers  int `json:"viewers"`
}

func (T *jax) cacheJaxResponse(key string, results []JaxGame) error {
	// Write to in-memory cache
	T.cachedTopGames = results

	return nil
}

func (T *jax) streamsFallback(err error, key string, out chan []JaxGame) {
	out <- T.cachedTopGames
}

// summaryRequest sends a request to the /stream/summary endpoint and
// returns the summary struct response.
func (T *jax) summaryRequest(ctx context.Context, opts url.Values) ([]JaxGame, error) {
	results := summary{}

	// Always summary requests group by game_id
	opts.Set("group-by", "rails.game_id")
	resp, err := T.performRequest(ctx, streamSummaryEndPoint, opts)
	if resp != nil {
		defer resp.Body.Close()
	}
	if err != nil {
		return results.Games, ErrJaxRequestFailed{Message: err.Error()}
	}

	if resp.StatusCode != http.StatusOK {
		return results.Games, ErrJaxRequestFailed{Message: resp.Status}
	}

	decoder := json.NewDecoder(resp.Body)
	if err := decoder.Decode(&results); err != nil {
		err := fmt.Errorf("streams: Error decoding Jax response: %v", err)
		return results.Games, err
	}

	return results.Games, nil
}

// LiveGames returns the ids of games currently being broadcasted, in a map format.
func (T *jax) LiveGames(ctx context.Context) (map[int]JaxGame, error) {
	timingStart := time.Now()

	opts := url.Values{
		"rails.directory_hidden": []string{"false"},
		"rails.category":         []string{"gaming"},
	}

	out := make(chan []JaxGame, 2)
	errors := hystrix.Go(cmdLiveGames, func() error {
		results, err := T.summaryRequest(ctx, opts)
		if err != nil {
			return err
		}

		T.cacheJaxResponse(cmdLiveGamesCacheKey, results)
		out <- results

		return nil
	}, func(err error) error {
		T.streamsFallback(err, cmdLiveGamesCacheKey, out)
		return nil
	})

	select {
	case results := <-out:
		games := make(map[int]JaxGame)
		for _, g := range results {
			games[g.ID] = g
		}
		T.stats.TimingDuration("streams.live_games.success", time.Since(timingStart), 0.1)
		return games, nil
	case err := <-errors:
		T.stats.TimingDuration("streams.live_games.error", time.Since(timingStart), 0.1)
		return nil, err
	}
}

// LiveGameInfo returns game info for certain games,
// specified by a comma separated list of game ids
func (T *jax) LiveGameInfo(ctx context.Context, ids string) (map[int]JaxGame, error) {
	timingStart := time.Now()

	opts := url.Values{
		"rails.directory_hidden": []string{"false"},
		"rails.category":         []string{"gaming"},
		"rails.game_id":          []string{ids},
	}

	out := make(chan []JaxGame, 2)
	errors := hystrix.Go(cmdLiveInfo, func() error {
		results, err := T.summaryRequest(ctx, opts)
		if err != nil {
			return err
		}

		T.cacheJaxResponse(cmdLiveInfoCacheKey, results)
		out <- results
		return nil
	}, func(err error) error {
		T.streamsFallback(err, cmdLiveInfoCacheKey, out)
		return nil
	})

	select {
	case results := <-out:
		games := make(map[int]JaxGame)
		for _, g := range results {
			games[g.ID] = g
		}
		T.stats.TimingDuration("streams.live_game_info.success", time.Since(timingStart), 0.1)
		return games, nil
	case err := <-errors:
		T.stats.TimingDuration("streams.live_game_info.error", time.Since(timingStart), 0.1)
		return nil, err
	}
}

// in returns whether an int is in a list of int.
func in(val int, list []int) bool {
	for _, v := range list {
		if val == v {
			return true
		}
	}
	return false
}

// TopGames returns an ordered list of games with live info.
func (T *jax) TopGames(ctx context.Context, ordering string) ([]JaxGame, error) {
	timingStart := time.Now()

	opts := url.Values{}

	// Default sorting is by number of viewers,
	// so only set if we want to sort by channels
	if ordering == "channels" {
		opts.Set("order", "channels")
	}

	out := make(chan []JaxGame, 2)
	errors := hystrix.Go(cmdTopGames, func() error {
		results, err := T.summaryRequest(ctx, opts)
		if err != nil {
			return err
		}

		T.cacheJaxResponse(cmdTopGamesCacheKey, results)
		out <- results
		return nil
	}, func(err error) error {
		T.streamsFallback(err, cmdTopGamesCacheKey, out)
		return nil
	})

	select {
	case results := <-out:
		T.stats.TimingDuration("streams.top_games.success", time.Since(timingStart), 0.1)
		return results, nil
	case err := <-errors:
		T.stats.TimingDuration("streams.top_games.error", time.Since(timingStart), 0.1)
		return nil, err
	}
}
