package api

import (
	"time"

	"code.justin.tv/feeds/feeds-common/entity"
)

// ExtractMessageRequest is the body for the POST /v1/maap/extraction/extract endpoint
type ExtractMessageRequest struct {
	// SenderID is the user sending the message.
	SenderID string `json:"sender_id"`
	// MessageText is the message body.
	MessageText string `json:"message_text"`
	// ContainerOwner is the entity that owns the container the message will be
	// published to, if any.
	// For example, chatrooms have owners, but whispers do not.
	ContainerOwner *entity.Entity `json:"container_owner_id"`

	// Optional requirements:

	// RequireNoBannedWords requires the message text to contain no banned words.
	RequireNoBannedWords bool `json:"require_no_banned_words"`
	// Max allowed character length in the message text before truncation.
	MaxCharacterLength *int `json:"max_character_length"`
}

// ExtractMessageResponse is the response for the POST /v1/maap/extraction/extract endpoint
type ExtractMessageResponse struct {
	Risk    MessageRisk    `json:"risk"`
	Content MessageContent `json:"content"`
	Sender  MessageSender  `json:"sender"`
}

// ListMessagesRequest is the body for the POST /v1/maap/messages/list endpoint.
type ListMessagesRequest struct {
	ContainerEntity         entity.Entity `json:"container_id"`
	Limit                   int           `json:"limit"`
	Cursor                  string        `json:"cursor"`
	SortAscending           bool          `json:"sort_ascending"`
	ReturnOnlyPendingReview bool          `json:"return_only_pending_review"`
}

// ListMessagesResponse is the response for the POST /v1/maap/messages/list endpoint.
type ListMessagesResponse struct {
	Messages []Message `json:"messages"`
	Cursor   string    `json:"cursor"`
}

// GetMessageRequest is the body for the POST /v1/maap/messages/get endpoint.
type GetMessageRequest struct {
	MessageID string `json:"message_id"`
}

// GetMessageResponse is the response for the POST /v1/maap/messages/get endpoint.
type GetMessageResponse struct {
	Message Message `json:"message"`
}

// CreateMessageRequest is the body for the POST /v1/maap/messages/create endpoint.
type CreateMessageRequest struct {
	ContainerEntity entity.Entity  `json:"container_id"`
	Content         MessageContent `json:"content"`
	Sender          MessageSender  `json:"sender"`
	Databag         Databag        `json:"databag"`

	// OrderByKey is an optional string that will be used for sorting/indexing
	// in the backend datastore. If not set, OrderByKey will default to the
	// message creation time in Unix nanoseconds.
	//
	// NOTE: It is the caller's responsibility to ensure this string is natively
	// sortable. For example, numbers are NOT sortable unless they are left-padded
	// to the same length (example: "400" > "1000", but "0400" < "1000").
	OrderByKey *string `json:"order_by_key"`
}

// CreateMessageResponse is the response for the POST /v1/maap/messages/create endpoint.
type CreateMessageResponse GetMessageResponse

// EditMessageRequest is the body for the POST /v1/maap/messages/edit endpoint.
type EditMessageRequest struct {
	MessageID string         `json:"message_id"`
	EditorID  string         `json:"editor_id"`
	Content   MessageContent `json:"content"`
}

// EditMessageResponse is the response for the POST /v1/maap/messages/edit endpoint.
type EditMessageResponse GetMessageResponse

// DeleteMessageRequest is the body for the POST /v1/maap/messages/delete endpoint.
type DeleteMessageRequest struct {
	MessageID string `json:"message_id"`
	DeleterID string `json:"deleter_id"`
}

// DeleteMessageResponse is the response for the POST /v1/maap/messages/delete endpoint.
type DeleteMessageResponse GetMessageResponse

// ApproveMessageRequest is the body for the POST /v1/maap/messages/approve endpoint.
type ApproveMessageRequest struct {
	MessageID  string `json:"message_id"`
	ApproverID string `json:"approver_id"`
}

// ApproveMessageResponse is the response for the POST /v1/maap/messages/approve endpoint.
type ApproveMessageResponse GetMessageResponse

// RejectMessageRequest is the body for the POST /v1/maap/messages/reject endpoint.
type RejectMessageRequest struct {
	MessageID  string `json:"message_id"`
	RejecterID string `json:"rejecter_id"`
}

// RejectMessageResponse is the response for the POST /v1/maap/messages/reject endpoint.
type RejectMessageResponse GetMessageResponse

// ListContainersByOwnerRequest is the body for the POST /v1/maap/containers/list endpoint.
type ListContainersByOwnerRequest struct {
	ContainerType  string        `json:"container_type"`
	ContainerOwner entity.Entity `json:"owner_id"`
	Limit          int           `json:"limit"`
	Cursor         string        `json:"cursor"`
}

// ListContainersByOwnerResponse is the body for the POST /v1/maap/containers/list endpoint.
type ListContainersByOwnerResponse struct {
	Containers []Container `json:"containers"`
	Cursor     string      `json:"cursor"`
}

// GetContainerRequest is the body for the POST /v1/maap/containers/get endpoint.
type GetContainerRequest struct {
	ContainerEntity entity.Entity `json:"container_id"`
}

// GetContainerResponse is the response for the POST /v1/maap/containers/get endpoint.
type GetContainerResponse struct {
	Container Container `json:"container"`
}

// CreateContainerRequest is the body for the POST /v1/maap/containers/create endpoint.
type CreateContainerRequest struct {
	ContainerEntity entity.Entity  `json:"container_id"`
	ContainerOwner  *entity.Entity `json:"owner_id"`
	RequireReview   string         `json:"require_review"`
	Databag         Databag        `json:"databag"`
	MemberUserIDs   []string       `json:"member_user_ids"`
}

// CreateContainerResponse is the response for the POST /v1/maap/containers/create endpoint.
type CreateContainerResponse GetContainerResponse

// UpdateContainerRequest is the body for the POST /v1/maap/containers/update endpoint.
type UpdateContainerRequest struct {
	ContainerEntity entity.Entity  `json:"container_id"`
	ContainerOwner  *entity.Entity `json:"owner_id"`
	RequireReview   *string        `json:"require_review"`
	Databag         *Databag       `json:"databag"`
}

// UpdateContainerResponse is the response for the POST /v1/maap/containers/update endpoint.
type UpdateContainerResponse GetContainerResponse

// ListContainerViewsByUserRequest is the body for the POST /v1/maap/container_views/list endpoint.
type ListContainerViewsByUserRequest struct {
	UserID        string `json:"user_id"`
	ContainerType string `json:"container_type"`
	Limit         int    `json:"limit"`
	Cursor        string `json:"cursor"`
	SortAscending bool   `json:"sort_ascending"`
}

// ListContainerViewsByUserResponse is the response for the POST /v1/maap/container_views/list endpoint.
type ListContainerViewsByUserResponse struct {
	ContainerViews []ContainerView `json:"container_views"`
	Cursor         string          `json:"cursor"`
}

// GetContainerViewRequest is the body for the POST /v1/maap/container_views/get endpoint.
type GetContainerViewRequest struct {
	UserID          string        `json:"user_id"`
	ContainerEntity entity.Entity `json:"container_id"`
}

// GetContainerViewResponse is the response for the POST /v1/maap/container_views/get endpoint.
type GetContainerViewResponse struct {
	ContainerView ContainerView `json:"container_view"`
}

// UpdateContainerViewRequest is the body for the POST /v1/maap/container_views/update endpoint.
type UpdateContainerViewRequest struct {
	UserID          string        `json:"user_id"`
	ContainerEntity entity.Entity `json:"container_id"`
	IsArchived      *bool         `json:"is_archived"`
	IsMuted         *bool         `json:"is_muted"`
	LastReadAt      *time.Time    `json:"last_read_at"`
}

// UpdateContainerViewResponse is the response for the POST /v1/maap/container_views/update endpoint.
type UpdateContainerViewResponse GetContainerViewResponse

// ListMembersRequest is the body for the POST /v1/maap/members/list endpoint.
type ListMembersRequest struct {
	ContainerEntity entity.Entity `json:"container_id"`
	Limit           int           `json:"limit"`
	Cursor          string        `json:"cursor"`
}

// ListMembersResponse is the response for the POST /v1/maap/members/list endpoint.
type ListMembersResponse struct {
	Members []MemberUser `json:"members"`
	Cursor  string       `json:"cursor"`
	Total   int          `json:"total"`
}

// GetMemberRequest is the body for the POST /v1/maap/members/get endpoint.
type GetMemberRequest struct {
	ContainerEntity entity.Entity `json:"container_id"`
	UserID          string        `json:"user_id"`
}

// GetMemberResponse is the response for the POST /v1/maap/members/get endpoint.
type GetMemberResponse struct {
	UserID          string        `json:"user_id"`
	ContainerEntity entity.Entity `json:"container_id"`
	JoinedAt        time.Time     `json:"joined_at"`
}

// AddMemberRequest is the body for the POST /v1/maap/members/add endpoint.
type AddMemberRequest struct {
	ContainerEntity entity.Entity `json:"container_id"`
	UserID          string        `json:"user_id"`
}

// AddMemberResponse is the response for the POST /v1/maap/members/add endpoint.
type AddMemberResponse GetMemberResponse

// RemoveMemberRequest is the body for the POST /v1/maap/members/remove endpoint.
type RemoveMemberRequest struct {
	ContainerEntity entity.Entity `json:"container_id"`
	UserID          string        `json:"user_id"`
}
