package spade

import (
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"golang.org/x/net/context"
)

// ErrMaxConcurrency is returned when there are outstanding Spade requests equaling
// the maximum concurrency set for a Client.
var ErrMaxConcurrency = errors.New("spade: max concurrency exceeded; ignoring tracking event")

// Client defines an interface to send tracking events to Spade
type Client interface {
	// TrackEvent sends an event tracking HTTP request to Spade.
	// It returns an error if there is a problem generating the request, if the
	// http.Client errors, the HTTP response is not a 200, or it has exceeded the
	// maximum concurrent request setting for the client.
	TrackEvent(ctx context.Context, event string, properties interface{}) error
}

var (
	defaultURL = &url.URL{
		Scheme: "http",
		Host:   "spade.twitch.tv",
		Path:   "/",
	}
	defaultURLValues = url.Values{}
	noopStatHook     = func(_ string, _ int, _ time.Duration) {}
)

// NewClient creates a new Spade client according to the configuration specified
// by its InitFuncs. If not specified via InitFunc, the default configuration
// options are used:
//
// BaseURL: "http://spade.twitch.tv/"
// MaxConcurrency: unlimited
func NewClient(opts ...InitFunc) (Client, error) {
	var init clientInit
	for _, opt := range opts {
		err := opt(&init)
		if err != nil {
			return nil, err
		}
	}

	if init.httpClientFunc != nil && init.httpClient != nil {
		return nil, errors.New("spade: cannot set both http client and http client func")
	}

	c := &clientImpl{}
	if init.hasBaseURL {
		c.baseURL = &init.baseURL
		c.baseURLValues = c.baseURL.Query()
	} else {
		c.baseURL = defaultURL
		c.baseURLValues = defaultURLValues
	}

	if init.httpClientFunc != nil {
		c.httpClientFunc = init.httpClientFunc
	} else if init.httpClient != nil {
		c.httpClientFunc = func(_ context.Context) *http.Client {
			return init.httpClient
		}
	} else {
		c.httpClientFunc = func(_ context.Context) *http.Client {
			return http.DefaultClient
		}
	}

	if init.maxConcurrency > 0 {
		c.sem = make(chan struct{}, init.maxConcurrency)
	}

	if init.statHook != nil {
		c.statHook = init.statHook
	} else {
		c.statHook = noopStatHook
	}

	return c, nil
}

// URLValues returns `url.Values` to be used in a Spade event tracking request.
// This can be used if you want more control over the HTTP request sent to Spade.
func URLValues(event string, properties interface{}) (url.Values, error) {
	jb, err := json.Marshal(payload{
		Event:      event,
		Properties: properties,
	})
	if err != nil {
		return nil, err
	}
	v := url.Values{}
	v.Add("data", base64.URLEncoding.EncodeToString(jb))
	return v, nil
}

type payload struct {
	Event      string      `json:"event"`
	Properties interface{} `json:"properties"`
}

type clientImpl struct {
	httpClientFunc func(context.Context) *http.Client
	baseURL        *url.URL
	baseURLValues  url.Values
	sem            chan struct{}
	statHook       StatHook
}

var _ Client = (*clientImpl)(nil)

func (c *clientImpl) TrackEvent(ctx context.Context, event string, properties interface{}) error {
	if c.sem != nil {
		select {
		case c.sem <- struct{}{}:
			defer func() {
				<-c.sem
			}()
		default:
			return ErrMaxConcurrency
		}
	}

	v, err := URLValues(event, properties)
	if err != nil {
		return err
	}
	req := http.Request{
		Method: "GET",
		URL:    createURL(c.baseURL, c.baseURLValues, v),
		Header: make(http.Header),
	}
	req.Header.Add("User-Agent", "code.justin.tv/common/spade-client/go")
	reqStart := time.Now()
	response, err := c.httpClientFunc(ctx).Do(&req)
	duration := time.Since(reqStart)
	if err != nil {
		c.statHook("track_event", 0, duration)
		return err
	}
	defer response.Body.Close()
	c.statHook("track_event", response.StatusCode, duration)
	if response.StatusCode != http.StatusNoContent {
		return fmt.Errorf("spade: unexpected response status %d", response.StatusCode)
	}
	return nil
}

func createURL(base *url.URL, baseValues url.Values, values url.Values) *url.URL {
	for k, v := range baseValues {
		if _, ok := values[k]; !ok {
			for _, x := range v {
				values.Add(k, x)
			}
		}
	}
	return &url.URL{
		Scheme:   base.Scheme,
		Opaque:   base.Opaque,
		User:     base.User,
		Host:     base.Host,
		Path:     base.Path,
		RawQuery: values.Encode(),
		Fragment: base.Fragment,
	}
}

// InitFuncs initialize Client implementations with configuration options.
type InitFunc func(*clientInit) error

type clientInit struct {
	maxConcurrency int

	httpClientFunc func(context.Context) *http.Client
	httpClient     *http.Client

	hasBaseURL bool
	baseURL    url.URL

	statHook StatHook
}

func InitHTTPClient(c *http.Client) InitFunc {
	return func(init *clientInit) error {
		init.httpClient = c
		return nil
	}
}

func InitHTTPClientFunc(fn func(context.Context) *http.Client) InitFunc {
	return func(init *clientInit) error {
		init.httpClientFunc = fn
		return nil
	}
}

// InitBaseURL defines the URL for Spade HTTP requests.
func InitBaseURL(base url.URL) InitFunc {
	return func(init *clientInit) error {
		if base.Host == "" {
			return fmt.Errorf("spade: missing host in url %q", base)
		}
		if base.Scheme == "" {
			base.Scheme = "http"
		}
		init.hasBaseURL = true
		init.baseURL = base
		return nil
	}
}

// InitMaxConcurrency defines the maximum number of outstanding HTTP requests
// a Client will allow to Spade. This protects the running process from over-allocating
// system resources (memory, network, CPU time) to Spade event tracking so that
// the running process can do higher-priority work.
func InitMaxConcurrency(max int) InitFunc {
	return func(init *clientInit) error {
		if max <= 0 {
			return fmt.Errorf("spade: max concurrency must be positive; received %d", max)
		}
		init.maxConcurrency = max
		return nil
	}
}

type StatHook func(name string, httpStatusCode int, d time.Duration)

// InitStatHook defines a callback to track Spade HTTP request stats
func InitStatHook(hook StatHook) InitFunc {
	return func(init *clientInit) error {
		init.statHook = hook
		return nil
	}
}
