package main

import (
	"bufio"
	"code.justin.tv/common/config"
	"code.justin.tv/discovery/insights/golib/insights"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path"
	"runtime"
)

func main() {
	config.Register(map[string]string{
		"index-name": "", // ex: insights-discovery
	})
	config.Parse()
	log.SetFlags(0)

	if len(os.Args) < 2 {
		showUsage()
		return
	}

	log.Println()
	log.Printf("Environment: %s", config.Environment())
	log.Printf("ElasticSearch Host: %s", esHost())
	log.Println()

	cmd := os.Args[1]
	if cmd == "template-get" {
		showTemplate()
	} else if cmd == "template-post" {
		postTemplate()
	} else if cmd == "index-get" {
		showIndex(config.MustResolve("index-name"))
	} else if cmd == "index-drop" {
		dropIndex(config.MustResolve("index-name"))
	} else {
		showUsage()
	}
}

func showIndex(indexName string) {
	url := fmt.Sprintf("http://%s/%s", esHost(), indexName)
	handleResponse(http.Get(url))
}

func dropIndex(indexName string) {
	// FIXME: add confirmation for extra safety here
	url := fmt.Sprintf("http://%s/%s", esHost(), indexName)
	request, err := http.NewRequest("DELETE", url, nil)
	if err != nil {
		log.Fatal(err)
	}
	handleResponse(http.DefaultClient.Do(request))
}

func showTemplate() {
	handleResponse(http.Get(templateUrl()))
}

func postTemplate() {
	_, scriptFilename, _, _ := runtime.Caller(1)
	insightsTemplatePath := path.Join(path.Dir(scriptFilename), "/", templateLocalPath())

	f, err := os.Open(insightsTemplatePath)
	defer f.Close()
	if err != nil {
		log.Fatal(err)
	}

	body := bufio.NewReader(f)
	handleResponse(http.Post(templateUrl(), "application/json", body))
}

func handleResponse(resp *http.Response, err error) {
	defer resp.Body.Close()
	if err != nil {
		log.Fatal(err)
	}

	contents, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Fatal(err)
	}

	fmt.Printf(string(contents))
	log.Println()
}

func templateUrl() string {
	return fmt.Sprintf("http://%s:80/_template/insights_template", esHost())
}

func templateLocalPath() string {
	return "terraform/scripts/insights-template.json"
}

func esHost() string {
	return insights.ElasticSearchHost(config.Environment())
}

func showUsage() {
	log.Printf("USAGE:")
	log.Println()
	log.Printf("  go run ./es-admin.go <command>")
	log.Println()
	log.Printf("ENVIRONMENT VARIABLES:")
	log.Println()
	log.Printf("  ENVIRONMENT      default is development")
	log.Println()
	log.Printf("COMMANDS:")
	log.Println()
	log.Printf("  template-get     Displays current insights template")
	log.Printf("  template-post    Posts insights template from %s", templateLocalPath())
	log.Printf("  index-get        Displays index (currently shows insights-discovery only)")
	log.Printf("  index-drop       Dropx an index, BE CAREFUL WITH THIS! (currently only drop insights-discovery)")
	log.Println()
}
