package insights

import (
	"code.justin.tv/discovery/insights/golib/awshttpauth"
	"encoding/json"
	"fmt"
	"github.com/abh/geoip"
	"github.com/afex/hystrix-go/hystrix"
	"gopkg.in/olivere/elastic.v2"
	"log"
	"net"
	"net/http"
	"os"
	"time"
)

const (
	elasticSearchDevHost  = "search-insights-dev-ucqzsusl3rzotgvyr7slt23u4q.us-west-2.es.amazonaws.com"
	elasticSearchProdHost = "search-insights-production-qruajilf3av7vq5iv7txfgdtdq.us-west-2.es.amazonaws.com"
	elasticSearchPort     = "80"
	hystrixCommand        = "log_event"
	maxConcurrentRequests = 180 // FIXME: I think this is dependent on the # of shards for the index, max is 200 but making smaller to be safe?
	bulkReadyElapsedTime  = 15 * time.Second
	bulkReadyCount        = 700
	bulkRequestsChanSize  = 1000
)

type Insights struct {
	AppName              string
	ElasticSearch        *elastic.Client
	bulkableRequestsChan chan elastic.BulkableRequest
	environment          string
	noop                 bool
	geoip                *geoip.GeoIP
	debug                bool
}

func ElasticSearchHost(environment string) string {
	if environment == "development" || environment == "staging" {
		return elasticSearchDevHost
	}
	return elasticSearchProdHost
}

func AwsAuthHttpClient() *http.Client {
	transport := &awshttpauth.Transport{}
	return &http.Client{Transport: transport}
}

func AuthlessHttpClient() *http.Client {
	return http.DefaultClient
}

func New(appName, environment string, httpClient *http.Client, debug bool) *Insights {
	var esClient *elastic.Client

	hystrix.ConfigureCommand(hystrixCommand, hystrix.CommandConfig{
		MaxConcurrentRequests: maxConcurrentRequests,
	})

	url := fmt.Sprintf("http://%s:%s", ElasticSearchHost(environment), elasticSearchPort)
	es, err := elastic.NewSimpleClient(elastic.SetURL(url), elastic.SetHttpClient(httpClient))
	if err != nil {
		log.Printf("[insights] Could not initialize ElasticSearch client: %s", err.Error())
	} else {
		esClient = es
	}

	geoipFile := "/usr/share/GeoIP/GeoIPCity.dat"
	var geoipDb *geoip.GeoIP
	gi, err := geoip.Open(geoipFile)
	if err != nil {
		log.Printf("[insights] Disabled GeoIP: %s", err.Error())
	} else {
		log.Printf("[insights] Enabled GeoIP.")
		geoipDb = gi
	}
	i := &Insights{
		AppName:              appName,
		ElasticSearch:        esClient,
		bulkableRequestsChan: make(chan elastic.BulkableRequest, bulkRequestsChanSize),
		environment:          environment,
		geoip:                geoipDb,
		debug:                debug,
	}

	go func() {
		bulkService := esClient.Bulk()
		batchTimeout := time.After(bulkReadyElapsedTime)

		for {
			select {
			case <-batchTimeout:
				i.doBulkRequest(bulkService)
			case bulkRequest := <-i.bulkableRequestsChan:
				bulkService.Add(bulkRequest)
				if bulkService.NumberOfActions() >= bulkReadyCount {
					if i.debug {
						log.Printf("[insights] Bulk size reached: %d. Doing request.", bulkService.NumberOfActions())
					}
					i.doBulkRequest(bulkService)
				}
			}
		}
	}()

	return i
}

func NewNoop() *Insights {
	return &Insights{noop: true}
}

func (I Insights) LogHttpRequestEvent(request *http.Request, eventName string, eventData map[string]interface{}) {
	level3ClientIp := request.Header.Get("X-Level3-Client-Ip")
	clientIp := request.Header.Get("Client-IP")
	forwardedForIp := request.Header.Get("X-Forwarded-For")

	var remoteHost string
	if len(level3ClientIp) > 0 {
		remoteHost = level3ClientIp
	} else if len(clientIp) > 0 {
		remoteHost = clientIp
	} else if len(forwardedForIp) > 0 {
		remoteHost = forwardedForIp
	} else {
		remoteHost, _, _ = net.SplitHostPort(request.RemoteAddr)
	}
	eventData["remote_ip"] = remoteHost

	if (I.geoip != nil) && (len(remoteHost) > 0) {
		record := I.geoip.GetRecord(remoteHost)
		if record != nil {
			eventData["geoip.location"] = fmt.Sprintf("%v, %v", record.Latitude, record.Longitude)
			eventData["geoip.country_name"] = record.CountryName
			eventData["geoip.country_code"] = record.CountryCode
			eventData["geoip.country_code3"] = record.CountryCode3
		}
	}

	request.ParseForm()
	headerJson, _ := json.Marshal(request.Header)
	eventData["request_header_s"] = string(headerJson)
	eventData["query_params_s"] = request.URL.RawQuery
	eventData["form_params_s"] = request.Form.Encode()
	eventData["user_agent_s"] = request.Header.Get("User-Agent")

	I.LogEvent(eventName, eventData)
}

func (I Insights) LogEvent(eventName string, eventData map[string]interface{}) {
	if I.noop {
		return
	}
	if I.ElasticSearch == nil {
		return
	}

	// FIXME: error check to ensure eventData is correct; or just a custom data type
	go func() {
		hostname, err := os.Hostname()
		if err != nil {
			hostname = ""
		}
		eventData["hostname_s"] = hostname

		// sets @timestamp, should be in ISO 8601 format
		// http://stackoverflow.com/questions/522251/whats-the-difference-between-iso-8601-and-rfc-3339-date-formats
		eventData["@timestamp"] = time.Now().Format(time.RFC3339)

		bulkRequest := elastic.NewBulkIndexRequest().Index(I.index()).Type(eventName).Doc(eventData)
		I.bulkableRequestsChan <- bulkRequest
	}()
}

func (I Insights) index() string {
	return fmt.Sprintf("insights-%s", I.AppName)
}

func (I Insights) doBulkRequest(bulkService *elastic.BulkService) {
	if bulkService.NumberOfActions() > 0 {
		hystrix.Do(hystrixCommand, func() error {
			_, err := bulkService.Do()
			if err != nil {
				if I.debug {
					log.Printf("[insights] LogEvent ERROR: %s", err.Error())
				}
				return err
			}
			if I.debug {
				log.Printf("[insights] LogEvent success.")
			}
			return nil
		}, func(err error) error {
			if I.debug {
				log.Printf("[insights] LogEvent failure fallback: %s", err.Error())
			}
			return nil
		})
	}
}
