package main

import (
	"strconv"
	"time"

	"os"

	"code.justin.tv/chat/pushy/client/events"
	"code.justin.tv/common/config"
	"code.justin.tv/common/hystrixstatsd"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/revenue/moneypenny/app/api"
	"code.justin.tv/revenue/moneypenny/app/auth"
	"code.justin.tv/revenue/moneypenny/app/clients/datastore"
	"code.justin.tv/revenue/moneypenny/app/clients/datawarehouse"
	"code.justin.tv/revenue/moneypenny/app/clients/guardian"
	"code.justin.tv/revenue/moneypenny/app/clients/payday"
	"code.justin.tv/revenue/moneypenny/app/clients/pushy"
	"code.justin.tv/revenue/moneypenny/app/util"
	"code.justin.tv/revenue/moneypenny/app/workers"
	conf "code.justin.tv/revenue/moneypenny/config"
	"code.justin.tv/revenue/moneypenny/lib/dscl"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodbstreams"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/heroku/rollrus"
	"sync"
)

const (
	HystrixSampleRate = 0.1
)

func init() {
	log.SetLevel(log.DebugLevel)
	log.Info("Starting moneypenny.....")

	//our application configuration
	config.Register(map[string]string{
		conf.AwsRegion:                    "us-west-2",
		conf.AddressInputARN:              "",
		conf.DlqSqs:                       "",
		conf.AddressVerificationNumWorker: "",
		conf.DlqNumWorker:                 "",
	})
}

func replaceString(p *string, replacement string) {
	if replacement != "" {
		*p = replacement
	}
}

func replaceBool(p *bool, replacement string) {
	if replacement != "" {
		v, err := strconv.ParseBool(replacement)
		//error no replacement
		if err != nil {
			panic(err)
		}
		*p = v
	}
}

func replaceInt(p *int, replacement string) {
	if replacement != "" {
		v, err := strconv.Atoi(replacement)
		//error no replacement
		if err != nil {
			panic(err)
		}
		*p = v
	}
}

func replaceConfigFromEnv(confObj *conf.Config) {
	//override configuration with command-line or Environment
	replaceString(&confObj.AwsRegion, config.Resolve(conf.AwsRegion))
	replaceString(&confObj.AddressSqsRegion, config.Resolve(conf.AddressSqsRegion))
	replaceString(&confObj.AddressInputArn, config.Resolve(conf.AddressInputARN))
	replaceString(&confObj.AddressVerificationResultSqsName, config.Resolve(conf.AddressVerificationResultSqs))
	replaceString(&confObj.TIMSSqsName, config.Resolve(conf.TIMSSqs))
	replaceString(&confObj.DlqSqsName, config.Resolve(conf.DlqSqs))
	replaceString(&confObj.TipaltiIPNSQSName, config.Resolve(conf.TipaltiIPNSQS))
	replaceInt(&confObj.AddressVerificationResultNumWorker, config.Resolve(conf.AddressVerificationNumWorker))
	replaceInt(&confObj.TIMSSqsNumWorker, config.Resolve(conf.TIMSNumWorker))
	replaceInt(&confObj.DlqSqsNumWorker, config.Resolve(conf.DlqNumWorker))
	replaceInt(&confObj.TipaltiIPNSQSNumWorker, config.Resolve(conf.TipaltiIPNNumWorker))
	replaceString(&confObj.TIMSEndpoint, config.Resolve(conf.TIMSEndpoint))
	replaceString(&confObj.TIMSCert, config.Resolve(conf.TIMSCert))
	replaceString(&confObj.PaydayEndpoint, config.Resolve(conf.PaydayEndpoint))
	replaceString(&confObj.TipaltiIframeEndpoint, config.Resolve(conf.TipaltiIframeEndpoint))
	replaceString(&confObj.TipaltiPayeeSoapEndpoint, config.Resolve(conf.TipaltiPayeeSoapEndpoint))
	replaceString(&confObj.TipaltiCert, config.Resolve(conf.TipaltiCert))
	replaceString(&confObj.CartmanCert, config.Resolve(conf.CartmanCert))
	replaceString(&confObj.LogOutput, config.Resolve(conf.LogOutput))
	replaceString(&confObj.KmsCustomerMasterKeyID, config.Resolve(conf.KmsCustomerMasterKeyID))
	replaceString(&confObj.AddressCustomerMasterKeyID, config.Resolve(conf.AddressCustomerMasterKeyID))
	replaceString(&confObj.DWConnParams, config.Resolve(conf.DWConnParams))
	replaceInt(&confObj.DWConnMaxAge, config.Resolve(conf.DWConnMaxAge))
	replaceString(&confObj.PushySnsArn, config.Resolve(conf.PushySNSARN))
	replaceString(&confObj.GuardianEndpoint, config.Resolve(conf.GuardianEndpoint))
	replaceString(&confObj.OnboardingSQSName, config.Resolve(conf.OnboardingSQS))
	replaceInt(&confObj.OnboardingSQSNumWorker, config.Resolve(conf.OnboardingSQSNumWorker))
	replaceString(&confObj.OnboardingStreamArn, config.Resolve(conf.OnboardingStreamArn))
	replaceBool(&confObj.UseInMemoryDB, config.Resolve(conf.UseInMemoryDB))
}

func main() {
	config.Parse()
	confObj, err := conf.LoadConfig()
	if err != nil {
		log.WithFields(log.Fields{
			"event":  "fatal_error_parsing_config",
			"detail": err.Error(),
		}).WithError(err).Fatalf("Fatal error parsing config")
	}
	replaceConfigFromEnv(confObj)

	log.WithFields(log.Fields{
		"event":  "config_object",
		"detail": fmt.Sprintf("%+v", confObj),
	}).Info()

	//Setup logger file
	file, warnErr := os.OpenFile(confObj.LogOutput, os.O_RDWR|os.O_CREATE|os.O_APPEND, 0666)
	if warnErr != nil {
		log.WithFields(log.Fields{
			"event":  "error_setting_log_output",
			"detail": warnErr.Error(),
		}).Warn()
	} else {
		defer file.Close()
		log.SetOutput(file)
		log.SetLevel(log.DebugLevel)
		log.SetFormatter(&log.TextFormatter{})
	}

	//Rollbar Token
	rollrus.SetupLogging(confObj.RollbarToken, conf.GetEnv())

	authHandler, err := auth.NewHandler(confObj.CartmanCert)
	if err != nil {
		log.WithFields(log.Fields{
			"event":  "fatal_error_creating_auth_handler",
			"detail": err.Error(),
		}).WithError(err).Fatalf("Fatal error creating auth handler")
	}

	awsSession, err := api.NewSession(confObj.AwsRegion)
	if err != nil {
		log.WithFields(log.Fields{
			"event":  "fatal_error_creating_aws_session",
			"detail": err.Error(),
		}).WithError(err).Fatalf("Fatal error creating aws session")
	}

	dao := datastore.NewDynamoDAO(awsSession, *confObj)
	callbacks := api.NewPayoutCallbacks(dao)
	statsdClients := config.Statsd()
	metricCollector.Registry.Register((&hystrixstatsd.StatsdCollectorFactory{
		Client:     statsdClients.NewSubStatter("hystrix"),
		SampleRate: HystrixSampleRate,
	}).Create)

	paydayClient, err := payday.NewPaydayClient(awsSession, *confObj, dao.UserAttributeDao)

	var pushyClient events.Publisher
	if confObj.PushySnsArn == "" {
		pushyClient = pushy.NewNoopPublisher()
	} else {
		pushyClient, err = events.NewPublisher(events.Config{
			DispatchSNSARN: confObj.PushySnsArn,
			Stats:          statsdClients,
			SNSClient:      sns.New(awsSession),
		})
	}

	guardianClient := guardian.NewGuardianClient(confObj.GuardianEndpoint)
	cache := util.NewCache(statsdClients)

	server := api.NewServer(&api.ServerContext{
		Config:          *confObj,
		Stats:           statsdClients,
		Dao:             dao,
		Callbacks:       callbacks,
		AuthHandler:     authHandler,
		PaydayClient:    paydayClient,
		PushyClient:     pushyClient,
		GuardianClient:  guardianClient,
		OnboardingCache: cache,
	})

	dw, err := datawarehouse.New(confObj.DWConnParams, confObj.DWConnMaxAge)
	if err != nil {
		log.WithFields(log.Fields{
			"event":  "fatal_error_creating_data_warehouse_connection",
			"detail": err.Error(),
		}).WithError(err).Fatalf("Fatal error cannot creating a connection to Data Warehouse")
	}

	timsSQSContext := workers.TIMSUpdateContext{
		PayoutEntityDao: dao.PayoutEntityDao,
		WorkflowDao:     dao.WorkflowDao,
		PayoutCallbacks: callbacks,
		Dw:              dw,
		PaydayClient:    paydayClient,
	}
	go workers.InitSQSWorkers(workers.CreateTIMSSQSProcessor(timsSQSContext), workers.SQSErrorHandling, statsdClients, &workers.SQSWorkerConfig{
		Region:        confObj.AwsRegion,
		QueueName:     confObj.TIMSSqsName,
		NumWorkers:    confObj.TIMSSqsNumWorker,
		MaxVisibility: time.Minute,
	})

	tipaltiSQSContext := workers.TipaltiIPNContext{
		PayoutEntityDao: dao.PayoutEntityDao,
		WorkflowDao:     dao.WorkflowDao,
		InvitationDao:   dao.AllOnboardingApplicationsDao,
		Config:          *confObj,
		Callbacks:       callbacks,
	}
	go workers.InitSQSWorkers(workers.CreateTipaltiIPNProcessor(tipaltiSQSContext), workers.SQSErrorHandling, statsdClients, &workers.SQSWorkerConfig{
		Region:        confObj.AwsRegion,
		QueueName:     confObj.TipaltiIPNSQSName,
		NumWorkers:    confObj.TipaltiIPNSQSNumWorker,
		MaxVisibility: time.Minute,
	})

	addressSQSContext := workers.AddressVerificationContext{
		PayoutEvent: dao.WorkflowDao,
		Callbacks:   callbacks,
	}
	go workers.InitSQSWorkers(workers.CreateAddressVerificationResultProcessor(addressSQSContext), workers.SQSErrorHandling, statsdClients, &workers.SQSWorkerConfig{
		Region:        confObj.AwsRegion,
		QueueName:     confObj.AddressVerificationResultSqsName,
		NumWorkers:    confObj.AddressVerificationResultNumWorker,
		MaxVisibility: time.Minute,
	})

	onboardingSQSContext := workers.OnboardingProcessorContext{
		PayoutEntityDao:         dao.PayoutEntityDao,
		OnboardingInvitationDao: dao.AllOnboardingApplicationsDao,
		WorkflowDao:             dao.WorkflowDao,
		Config:                  *confObj,
	}
	go workers.InitSQSWorkers(workers.CreateOnboardingProcessor(onboardingSQSContext), workers.SQSErrorHandling, statsdClients, &workers.SQSWorkerConfig{
		Region:        confObj.AwsRegion,
		QueueName:     confObj.OnboardingSQSName,
		NumWorkers:    confObj.OnboardingSQSNumWorker,
		MaxVisibility: time.Minute,
	})

	//Single DLQ for all SQS
	dlqContext := workers.PayoutDLQContext{
		TIMS:       timsSQSContext,
		Address:    addressSQSContext,
		Tipalti:    tipaltiSQSContext,
		Onboarding: onboardingSQSContext,
		Dao:        dao.DlqDao,
	}
	go workers.InitSQSWorkers(workers.CreateDynamoDLQProcessor(dlqContext), workers.SQSErrorHandling, statsdClients, &workers.SQSWorkerConfig{
		Region:        confObj.AwsRegion,
		QueueName:     confObj.DlqSqsName,
		NumWorkers:    confObj.DlqSqsNumWorker,
		MaxVisibility: time.Minute,
	})

	cacheUpdateLock := &sync.Mutex{}
	initCacheUpdater(awsSession, cache, cacheUpdateLock, confObj.OnboardingStreamArn)

	//populate in-memory database
	if confObj.UseInMemoryDB {
		log.Info("Scan all user status")
		records, err := dao.OnboardingStatusDao.RetrieveAllOnboardingStatusCache()
		if err != nil {
			log.WithError(err).Fatalf("Fatal error cannot populate data for in-memory database")
		}
		log.Info("Populate Cache")
		SyncCache(cacheUpdateLock, cache, records)
		log.Info("Update Records:", len(records), ", Current Cache Size:", cache.Size())
	}

	log.Fatal(twitchhttp.ListenAndServe(server))
}

func initCacheUpdater(awsSession *session.Session, cache *util.Cache, cacheUpdateLock *sync.Mutex, streamArn string) {
	streamWorkerInit := make(chan bool)
	//cache updater
	cacheUpdater := &workers.CacheUpdater{Cache: cache, UpdateLock: cacheUpdateLock}
	dynamoStream := dscl.New(dynamodbstreams.New(awsSession), streamArn, dscl.PositionLatest, cacheUpdater.Process)
	go dynamoStream.Start(streamWorkerInit)

	select {
	case <-time.After(60 * time.Second):
		panic("Timeout while waiting for dynamostream initialize")
	case <-streamWorkerInit:
		log.Debug("DyanmoDB worker is started")
	}

}

func SyncCache(lock *sync.Mutex, cache *util.Cache, records []datastore.OnboardingStatusRecord) {
	lock.Lock()
	defer lock.Unlock()

	for i := range records {
		channelID := records[i].ChannelID
		if !cache.Exist(channelID) {
			cache.Put(channelID, &records[i])
		}
	}
}
