package frontend

import (
	"fmt"
	"net/http"
	"time"

	"goji.io/pat"

	"goji.io"

	"code.justin.tv/foundation/twitchserver"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/common/stats"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	esTimeout = 5 * time.Second
)

// frontend registers the queryable endpoints
type frontend struct {
	*goji.Mux
	db     db.JaxReader
	writer db.JaxWriter

	Stats statsd.Statter
}

// New creates a new frontend to handle the different /stream/ endpoints.
func New(conf *config.Config, reader db.JaxReader, writer db.JaxWriter) *frontend {
	s := twitchserver.NewServer()

	fe := &frontend{
		Mux:    s,
		db:     reader,
		writer: writer,
		Stats:  stats.InitStatsd(conf),
	}

	s.HandleFunc(pat.Post("/stream/event"), routeWrapper("stream_event", fe.handleStreamEvent))
	s.HandleFunc(pat.Delete("/stream/event"), routeWrapper("stream_event", fe.handleStreamEvent))

	s.HandleFunc(pat.Get("/stream"), routeWrapper("get_stream", fe.handleGet))
	s.HandleFunc(pat.Get("/stream/list"), routeWrapper("get_streams_list", fe.handleStreamList))
	s.HandleFunc(pat.Post("/stream/list"), routeWrapper("get_streams_list", fe.handleStreamList))
	s.HandleFunc(pat.Get("/streams"), routeWrapper("streams", fe.handleStreams))
	s.HandleFunc(pat.Post("/streams"), routeWrapper("streams", fe.handleStreams))
	s.HandleFunc(pat.Get("/stream/summary"), routeWrapper("summary", fe.handleSummary))

	s.HandleFunc(pat.Get("/health"), routeWrapper("health", fe.handleHealth))

	// TODO: remove this after https://git-aws.internal.justin.tv/web/jax/pull/186 is merged
	s.HandleFunc(pat.Get("/debug/health"), routeWrapper("health", fe.handleHealth))

	return fe
}

func routeWrapper(name string, handler func(http.ResponseWriter, *http.Request)) func(http.ResponseWriter, *http.Request) {
	return func(w http.ResponseWriter, r *http.Request) {
		handler(log.NewLoggingResponseWriter(w, name), r)
	}
}

// handleHealth is the handler for the /debug/health endpoint. It is used to
// monitor the health of this process.
func (fe *frontend) handleHealth(w http.ResponseWriter, r *http.Request) {
	err := fe.checkBackendHealth()
	if err != nil {
		log.Reportf("%v", err)
		w.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(w, "%s", "ElasticSearch did not respond in time")
		log.ReportStats("health", time.Now(), http.StatusInternalServerError)
	} else {
		w.WriteHeader(http.StatusOK)
		fmt.Fprintf(w, "%s", "OK")
		log.ReportStats("health", time.Now(), http.StatusOK)
	}
}

// checkBackendHealth checks whether the local ElasticSearch instance is reachable in time
func (fe *frontend) checkBackendHealth() error {
	start := time.Now()

	_, err := fe.db.Search(nil, query.SearchQuery{
		Limit: 1,
	})

	end := time.Now()
	if err != nil {
		return fmt.Errorf("could not query elasticsearch: %s", err)
	} else if end.Sub(start) > esTimeout {
		return fmt.Errorf("health check query to elasticsearch took too long: > %v", esTimeout)
	}
	return nil
}
