package frontend

import (
	"bytes"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/mocks"
	"code.justin.tv/web/jax/db/query"

	"github.com/cactus/go-statsd-client/statsd"
	. "github.com/smartystreets/goconvey/convey"
)

func TestHandleGet(t *testing.T) {
	reader := &mocks.JaxReader{}
	router := &frontend{db: reader, Stats: &statsd.NoopClient{}}
	log.Init(&config.Config{RollbarToken: "322"})

	testChannel := "sldkfj"
	testField := "hello"
	testFilters := []query.Filter{query.ExistsFieldFilter("usher.id")}
	testStream := db.ChannelResult{
		Channel: testChannel,
		Properties: map[string]interface{}{
			testField: "test",
		},
	}

	Convey("checks channel parameter", t, func() {
		w := makeTestRequest("GET", router.handleGet, url.Values{"channel": []string{"~@#!@"}}, []byte{})

		So(w.Code, ShouldEqual, http.StatusOK)
	})

	Convey("returns result for id", t, func() {
		var fields []string
		reader.On("GetByID", testChannel, fields, testFilters).Return(&db.ResultSet{
			Total: 1,
			Hits: []db.ChannelResult{
				testStream,
			},
		}, nil).Once()

		v := url.Values{}
		v.Set("id", testChannel)
		w := makeTestRequest("GET", router.handleGet, v, []byte{})

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldContainSubstring, "max-age=")
		So(w.Body.String(), ShouldContainSubstring, testChannel)
		So(w.Body.String(), ShouldContainSubstring, testField)
	})

	Convey("returns result", t, func() {
		var fields []string
		reader.On("Get", testChannel, fields, testFilters).Return(&db.ResultSet{
			Total: 1,
			Hits: []db.ChannelResult{
				testStream,
			},
		}, nil).Once()

		v := url.Values{}
		v.Set("channel", testChannel)
		w := makeTestRequest("GET", router.handleGet, v, []byte{})

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldContainSubstring, "max-age=")
		So(w.Body.String(), ShouldContainSubstring, testChannel)
		So(w.Body.String(), ShouldContainSubstring, testField)
	})
}

func makeTestRequest(
	method string,
	h func(http.ResponseWriter, *http.Request),
	params url.Values,
	body []byte,
) *httptest.ResponseRecorder {
	r, _ := http.NewRequest(method, "/", bytes.NewReader(body))
	r.URL.RawQuery = params.Encode()

	w := httptest.NewRecorder()
	h(w, r)

	return w
}
