package frontend

import (
	"encoding/json"
	"net/http"
	"net/url"
	"testing"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db/mocks"

	"github.com/cactus/go-statsd-client/statsd"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestHandleStreamEvent(t *testing.T) {
	writer := &mocks.JaxWriter{}
	router := &frontend{writer: writer, Stats: &statsd.NoopClient{}}
	log.Init(&config.Config{RollbarToken: "322"})

	testChannel := "sldkfj"

	Convey("must be post or delete", t, func() {
		w := makeTestRequest("GET", router.handleStreamEvent, url.Values{}, []byte{})

		So(w.Code, ShouldEqual, http.StatusMethodNotAllowed)
	})

	Convey("malformed input", t, func() {
		w := makeTestRequest("POST", router.handleStreamEvent, url.Values{}, []byte("lsjfkld"))

		So(w.Code, ShouldEqual, http.StatusBadRequest)
	})

	Convey("updates for POST", t, func() {
		writer.On("Update", testChannel, mock.Anything).Return(nil)

		input := map[string]string{
			"channel": testChannel,
		}
		b, _ := json.Marshal(input)
		w := makeTestRequest("POST", router.handleStreamEvent, url.Values{}, b)

		So(w.Code, ShouldEqual, http.StatusOK)
	})

	Convey("deletes for DELETE", t, func() {
		writer.On("Delete", testChannel).Return(nil)
		writer.On("DeleteLegacy", testChannel).Return(nil)

		input := map[string]string{
			"channel": testChannel,
		}
		b, _ := json.Marshal(input)
		w := makeTestRequest("DELETE", router.handleStreamEvent, url.Values{}, b)

		So(w.Code, ShouldEqual, http.StatusOK)
	})
}
