package frontend

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"

	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const defaultCacheTimeSeconds = 300

// handleStreams is the handler for the /streams endpoint. It processes the URL
// parameters and body, sends the request to the backend and returns the result.
func (fe *frontend) handleStreams(w http.ResponseWriter, r *http.Request) {
	q := r.URL.Query()
	sq := query.SearchQuery{}
	if len(q.Get("sort")) > 0 {
		sq.SortFields = []string{q.Get("sort")}
	}

	if len(q.Get("locale")) > 0 {
		sq.Locale = strings.ToLower(q.Get("locale"))
	}

	sq.Limit, sq.Offset, sq.Fields, sq.Filters = parseQueryParams(q)

	if r.Method == "POST" {
		// Override fields and filters from POST body
		// TODO: deprecate this section
		body, err := ioutil.ReadAll(r.Body)
		if err != nil {
			serveError(w, r, http.StatusBadRequest, "error reading request body: %v", err)
			return
		}

		b := map[string]interface{}{}
		if len(body) > 0 {
			// We lowercase the incoming body to have case-insensitive searches.
			dec := json.NewDecoder(bytes.NewReader(bytes.ToLower(body)))
			dec.UseNumber()
			if err := dec.Decode(&b); err != nil {
				serveError(w, r, http.StatusBadRequest, "error parsing json: %v", err)
				return
			}
		}

		if arr, ok := b["fields"].([]interface{}); ok {
			if sq.Fields == nil {
				sq.Fields = []string{}
			}
			for _, v := range arr {
				if field, ok := v.(string); ok {
					sq.Fields = append(sq.Fields, field)
				}
			}
		}

		if filters, ok := b["filter"].(map[string]interface{}); ok {
			sq.Filters = append(sq.Filters, query.BodyToFilters(filters)...)
		}
	}

	resp, dbErr := fe.db.Search(nil, sq)
	if dbErr != nil {
		serveDbError(w, r, dbErr)
		return
	}
	w.Header().Set("Content-Type", contentType)
	if r.Method == "GET" {
		seconds := requestCacheTime(q)
		if seconds == 0 {
			w.Header().Set("Cache-Control", "no-cache")
		} else {
			w.Header().Set("Cache-Control", fmt.Sprintf("max-age=%d", seconds))
		}
	}
	w.WriteHeader(http.StatusOK)

	fe.Stats.Inc("response.streams.num_streams", int64(len(resp.Hits)), 0.1)

	res, jsonErr := json.Marshal(resp)
	if jsonErr != nil {
		err := db.NewJaxDbError(jsonErr.Error())
		serveDbError(w, r, err)
		return
	}
	fmt.Fprintf(w, "%s", res)
}

var noCacheGameIDs = map[string]bool{
	"488552": true,
	"21779":  true,
	"138585": true,
}

var noCacheGameNames = map[string]bool{
	"overwatch":                     true,
	"league of legends":             true,
	"hearthstone":                   true,
	"playerunknown's battlegrounds": true,
}

func requestCacheTime(q url.Values) int {
	if q.Get("rails.game_id") != "" {
		if noCacheGameIDs[q.Get("rails.game_id")] {
			return 0
		}
	}

	if q.Get("rails.meta_game") != "" {
		if noCacheGameNames[strings.ToLower(q.Get("rails.meta_game"))] {
			return 0
		}
	}

	return defaultCacheTimeSeconds
}
