package frontend

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"testing"

	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/mocks"
	"code.justin.tv/web/jax/db/query"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/mock"

	. "github.com/smartystreets/goconvey/convey"
)

var (
	TestSet = db.ResultSet{
		ScrollID: "",
		Total:    2,
		Hits: []db.ChannelResult{
			db.ChannelResult{
				Channel: "twitchplayspokemon",
				Properties: map[string]interface{}{
					"channel":       "twitchplayspokemon",
					"channel_count": 1234567,
				},
			},
			db.ChannelResult{
				Channel: "testchannel",
				Properties: map[string]interface{}{
					"channel":       "testchannel",
					"channel_count": 7654321,
				},
			},
		},
	}

	TestBody = []byte(`
        { "filter" : { "eq" : { "rails.channel" : "testchannel" } } }
    `)

	InvalidTestBody = []byte(`
        { "filter" :  "eq" : { "rails.channel" : "testchannel" } } }
    `)
)

func TestHandleStreams(t *testing.T) {
	reader := &mocks.JaxReader{}
	router := &frontend{db: reader, Stats: &statsd.NoopClient{}}

	Convey("POST a valid search body", t, func() {
		reader.On("Search", mock.Anything, mock.Anything).Return(&TestSet, nil).Once()

		w := makeTestRequest("POST", router.handleStreams, url.Values{}, TestBody)

		exp, _ := json.Marshal(TestSet)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		So(w.Body.String(), ShouldEqual, string(exp))
	})

	Convey("POST an invalid search body", t, func() {
		reader.On("Search", mock.Anything, mock.Anything).Return(&TestSet, nil).Once()

		w := makeTestRequest("POST", router.handleStreams, url.Values{}, InvalidTestBody)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
	})

	Convey("GET", t, func() {
		reader.On("Search", mock.Anything, mock.Anything).Return(&TestSet, nil).Once()

		w := makeTestRequest("GET", router.handleStreams, url.Values{}, nil)

		exp, _ := json.Marshal(TestSet)
		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldContainSubstring, fmt.Sprintf("max-age=%d", defaultCacheTimeSeconds))
		So(w.Body.String(), ShouldEqual, string(exp))
	})

	Convey("sort param", t, func() {
		sort := "sdbnlasd"
		u := url.Values{}
		u.Set("sort", sort)

		reader.On("Search", mock.Anything, mock.MatchedBy(func(q query.SearchQuery) bool {
			return len(q.SortFields) == 1 && q.SortFields[0] == sort
		})).Return(&TestSet, nil).Once()

		w := makeTestRequest("GET", router.handleStreams, u, nil)

		exp, _ := json.Marshal(TestSet)
		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldContainSubstring, fmt.Sprintf("max-age=%d", defaultCacheTimeSeconds))
		So(w.Body.String(), ShouldEqual, string(exp))
	})

	Convey("uncached for certain game", t, func() {
		u := url.Values{}
		u.Set("rails.game_id", "21779")

		reader.On("Search", mock.Anything, mock.Anything).Return(&TestSet, nil).Once()

		w := makeTestRequest("GET", router.handleStreams, u, nil)

		exp, _ := json.Marshal(TestSet)
		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldContainSubstring, "no-cache")
		So(w.Body.String(), ShouldEqual, string(exp))
	})

	Convey("mixed fields between url and body for POST", t, func() {
		fields := []string{"test", "test2"}
		bodyField := "test-ldksjf"

		body, _ := json.Marshal(map[string]interface{}{
			"fields": []string{bodyField},
		})

		u := url.Values{}
		u.Set("fields", strings.Join(fields, ","))

		reader.On("Search", mock.Anything, mock.MatchedBy(func(q query.SearchQuery) bool {
			return len(q.Fields) == len(fields)+1
		})).Return(&TestSet, nil).Once()

		w := makeTestRequest("POST", router.handleStreams, u, body)

		exp, _ := json.Marshal(TestSet)
		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		So(w.Body.String(), ShouldEqual, string(exp))
	})

	Convey("mixed filters between url and body for POST", t, func() {
		testField := "asdlj"
		testValue := "ertydf"
		testBroadcaster := "sdfsdf"

		body, _ := json.Marshal(map[string]interface{}{
			"filter": map[string]interface{}{
				"eq": map[string]interface{}{
					testField: testValue,
				},
			},
		})

		u := url.Values{}
		u.Set("usher.broadcaster", testBroadcaster)

		reader.On("Search", mock.Anything, mock.MatchedBy(func(q query.SearchQuery) bool {
			// Both filters must be there plus the usher.id filter
			if len(q.Filters) != 3 {
				return false
			}

			// Correct properties should satisfy the filters
			for _, f := range q.Filters {
				if !f.Valid(map[string]interface{}{
					testField:           testValue,
					"usher.id":          "sdlfkjsdf",
					"usher.broadcaster": testBroadcaster,
				}) {
					return false
				}
			}

			// Incorrect properties should not satisfy the filters
			valid := true
			for _, f := range q.Filters {
				if !f.Valid(map[string]interface{}{
					testField:           "dfglkjdfgkljerilogndfkl",
					"usher.id":          "sfsdf",
					"usher.broadcaster": testBroadcaster,
				}) {
					valid = false
				}

				if !f.Valid(map[string]interface{}{
					testField:           testValue,
					"usher.id":          "Sfsdf",
					"usher.broadcaster": "sgjkldsfk",
				}) {
					valid = false
				}

				if !f.Valid(map[string]interface{}{
					testField:           testValue,
					"usher.broadcaster": testBroadcaster,
				}) {
					valid = false
				}
			}
			if valid {
				return false
			}

			return true
		})).Return(&TestSet, nil).Once()

		w := makeTestRequest("POST", router.handleStreams, u, body)

		exp, _ := json.Marshal(TestSet)
		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Body.String(), ShouldEqual, string(exp))
	})
}
