package frontend

import (
	"encoding/json"
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const contentType = "application/json; charset=utf-8"

type SummaryResponse struct {
	Results []db.Aggregate
	Total   int
	Type    string

	includeSortValue bool
}

func (s *SummaryResponse) MarshalJSON() ([]byte, error) {
	results := []map[string]interface{}{}
	for _, r := range s.Results {
		result := map[string]interface{}{}
		if s.Type != "" {
			result[s.Type] = r.Value
		}
		result["viewers"] = r.Viewers
		result["channels"] = r.Channels
		if s.includeSortValue && r.SortValue != nil {
			result["sort_value"] = r.SortValue
		}
		results = append(results, result)
	}

	response := map[string]interface{}{}
	response["results"] = results
	return json.Marshal(response)
}

func (fe *frontend) handleSummary(w http.ResponseWriter, r *http.Request) {
	q := r.URL.Query()

	aq := query.AggregationQuery{}
	if len(q.Get("group-by")) > 0 {
		aq.GroupBy = q.Get("group-by")
	} else {
		aq.GroupBy = "_index"
	}

	if len(q.Get("locale")) > 0 {
		aq.Locale = strings.ToLower(q.Get("locale"))
	}

	if len(q.Get("sort")) > 0 {
		aq.Sort = q.Get("sort")
	}

	_, _, _, filters := parseQueryParams(q)

	aq.Filters = filters
	if lim, err := strconv.Atoi(q.Get("size")); err == nil && lim > 0 {
		aq.Limit = lim
	}

	// Only include the sort value if we actually have a custom sort
	includeSort := q.Get("include-sort-value") == "true" && aq.Sort != ""

	resp, dbErr := fe.aggregate(aq, q.Get("group-by"), includeSort)

	if dbErr != nil {
		serveDbError(w, r, dbErr)
		return
	}
	w.Header().Set("Content-Type", contentType)
	w.WriteHeader(http.StatusOK)
	w.Write(resp)
}

func (fe *frontend) aggregate(q query.AggregationQuery, key string, includeSortValue bool) ([]byte, *db.JaxDbError) {
	aggs, err := fe.db.Aggregate(q)

	if err != nil {
		return nil, err
	}

	response := SummaryResponse{
		Total:            0,
		Results:          aggs,
		Type:             key,
		includeSortValue: includeSortValue,
	}
	res, e := response.MarshalJSON()

	if e != nil {
		err = db.NewJaxDbError(e.Error())
		return nil, err
	}

	return res, nil
}
