package frontend

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"code.justin.tv/web/jax/common/log"
	"code.justin.tv/web/jax/db"
	"code.justin.tv/web/jax/db/query"
)

const (
	defaultLimit = 20
)

// parseQueryParams returns limit, offset, fields, and filters from url parameters
func parseQueryParams(q url.Values) (int, int, []string, []query.Filter) {
	fields := strings.Split(strings.ToLower(q.Get("fields")), ",")
	if len(q.Get("fields")) == 0 {
		fields = nil
	}

	var limit, offset int
	limit = defaultLimit
	filters := []query.Filter{query.ExistsFieldFilter("usher.id")}
	for key, values := range q {
		value := strings.ToLower(values[0])
		if len(value) == 0 {
			continue
		}

		switch key {
		case "usher.avc_level",
			"usher.avc_profile",
			"rails.broadcaster_language",
			"usher.broadcaster",
			"usher.client_id",
			"usher.configuration",
			"rails.language",
			"rails.game_id",
			"zuma.community_id", // deprecated – moving to community_ids
			"zuma.community_ids",
			"playstation.sce_platform",
			"leagueoflegends.champion_id",
			"leagueoflegends.summoner_rank_numeric",
			"csgo.map",
			"csgo.skill",
			"hearthstone.broadcaster_hero_name",
			"hearthstone.broadcaster_hero_class",
			"hearthstone.game_mode",
			"pubg.game_mode",
			"overwatch.broadcaster_character":
			filters = append(filters, query.StringTermsFilter(key, strings.Split(value, ",")))
		case "pubg.player_alive_count_max":
			filters = append(filters, query.CompareFilter("lte", "pubg.player_alive_count", value))
		case "pubg.player_alive_count_min":
			filters = append(filters, query.CompareFilter("gte", "pubg.player_alive_count", value))
		case "usher.hls", "partnerships.partner_program", "rails.directory_hidden", "rails.meta_game":
			filters = append(filters, query.TermFilter(key, value))
		case "usher.max_height":
			maxHeight, err := strconv.Atoi(value)
			if err == nil {
				filters = append(filters, query.CompareFilter("lte", key, maxHeight))
			}
		case "stream_type":
			filters = append(filters, query.StreamTypeFilter(value))
		case "broadcast_platform":
			filters = append(filters, query.BroadcastPlatformFilter(value))
		case "xbox_heartbeat":
			if value == "true" {
				filters = append(filters, query.XboxHeartbeatFilter())
			}
		case "from":
			offset, _ = strconv.Atoi(value)
		case "size":
			limit, _ = strconv.Atoi(value)
			if limit == 0 {
				limit = defaultLimit
			}
		case "exists_fields":
			for _, field := range strings.Split(value, ",") {
				filters = append(filters, query.ExistsFieldFilter(field))
			}
		}
	}

	return limit, offset, fields, filters
}

// badSearchRequest is a utility function which writes the response header and
// logs the encountered error in the logger.
func badRequest(w http.ResponseWriter, status int, message string, err interface{}) {
	w.WriteHeader(status)
	log.Reportf(message, err)
	return
}

func serveError(w http.ResponseWriter, r *http.Request, status int, message string, err interface{}) {
	w.Header().Set("Content-Type", contentType)
	w.WriteHeader(status)
	message = fmt.Sprintf(message, err)
	m := map[string]string{"error": message}
	j, err := json.Marshal(m)
	if err == nil {
		fmt.Fprintf(w, string(j))
	}

	if status >= 500 {
		log.RequestError(r, status, fmt.Errorf("%s", message))
	}
	return
}

func serveDbError(w http.ResponseWriter, r *http.Request, dbErr *db.JaxDbError) {
	errMsg := "failed to process query: %v"
	if dbErr.Code == http.StatusInternalServerError {
		serveError(w, r, http.StatusInternalServerError, errMsg, dbErr)
	} else {
		serveError(w, r, http.StatusBadRequest, errMsg, dbErr.Code)
	}
}

func serveJsonError(w http.ResponseWriter, r *http.Request, json, err interface{}) {
	w.WriteHeader(http.StatusInternalServerError)
	log.Reportf("failed to marshal: %v ... %+v", json, err)
	return
}
