package frontend

import (
	"encoding/json"
	"net/url"
	"strconv"
	"strings"
	"testing"

	"code.justin.tv/web/jax/db/query"

	. "github.com/smartystreets/goconvey/convey"
)

func TestParse(t *testing.T) {
	testFields := []string{"foo", "bar"}

	Convey("parses limit", t, func() {
		testLimit := 5

		v := url.Values{}
		v.Set("size", strconv.Itoa(testLimit))
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, testLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("defaults limit for badly formatted limit", t, func() {
		v := url.Values{}
		v.Set("size", "sdfd")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("parses offset", t, func() {
		testOffset := 10

		v := url.Values{}
		v.Set("from", strconv.Itoa(testOffset))
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, testOffset)
		So(fields, ShouldBeNil)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("defaults to no offset for bad formatting", t, func() {
		v := url.Values{}
		v.Set("from", "sdgsrf")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("parses fields", t, func() {
		v := url.Values{}
		v.Set("fields", strings.Join(testFields, ","))
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldResemble, testFields)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("blank field parameter results in nil fields", t, func() {
		v := url.Values{}
		v.Set("fields", "")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)
		So(filters[0], ShouldResemble, query.ExistsFieldFilter("usher.id"))
	})

	Convey("parses terms filters", t, func() {
		validParams := map[string][]string{
			"usher.avc_level":            []string{"dfh"},
			"usher.avc_profile":          []string{"ghj"},
			"rails.broadcaster_language": []string{"345r", "345f66"},
			"usher.broadcaster":          []string{"dfg"},
			"usher.client_id":            []string{"fgh", "rtuy"},
			"usher.configuration":        []string{"lsdkcxvfj"},
			"rails.meta_game":            []string{"lsdft6kfj,sdflkj"},
			"rails.language":             []string{"dfg", "asdgs"},
			"playstation.sce_platform":   []string{"lfguhsdkfj", "asdsjkh"},
		}

		v := url.Values{}
		for k, vals := range validParams {
			v.Set(k, strings.Join(vals, ","))
		}
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		for k, v := range validParams {
			b2, err2 := json.Marshal(query.StringTermsFilter(k, v))
			So(err2, ShouldBeNil)
			So(string(b), ShouldContainSubstring, string(b2))
		}
	})

	Convey("parses term filters", t, func() {
		validParams := map[string]string{
			"usher.hls":                    "sdfsd",
			"partnerships.partner_program": "true",
			"rails.directory_hidden":       "sdtdr",
			"rails.game_id":                "hdfgf",
		}

		v := url.Values{}
		for k, val := range validParams {
			v.Set(k, val)
		}
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		for k, v := range validParams {
			b2, err2 := json.Marshal(query.TermFilter(k, v))
			So(err2, ShouldBeNil)
			So(string(b), ShouldContainSubstring, string(b2))
		}
	})

	Convey("parses max_height filter", t, func() {
		maxHeight := 23908
		v := url.Values{}
		v.Set("usher.max_height", strconv.Itoa(maxHeight))
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		b2, err2 := json.Marshal(query.CompareFilter("lte", "usher.max_height", maxHeight))
		So(err2, ShouldBeNil)
		So(string(b), ShouldContainSubstring, string(b2))
	})

	Convey("parses stream_type playlist", t, func() {
		v := url.Values{}
		v.Set("stream_type", "playlist")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		b2, err2 := json.Marshal(query.StreamTypeFilter("playlist"))
		So(err2, ShouldBeNil)
		So(string(b), ShouldContainSubstring, string(b2))
	})

	Convey("parses stream_type live", t, func() {
		v := url.Values{}
		v.Set("stream_type", "live")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		b2, err2 := json.Marshal(query.StreamTypeFilter("live"))
		So(err2, ShouldBeNil)
		So(string(b), ShouldContainSubstring, string(b2))
	})

	Convey("parses xbox_heartbeat", t, func() {
		v := url.Values{}
		v.Set("xbox_heartbeat", "true")
		limit, offset, fields, filters := parseQueryParams(v)

		So(limit, ShouldEqual, defaultLimit)
		So(offset, ShouldEqual, 0)
		So(fields, ShouldBeNil)

		b, err := json.Marshal(filters)
		So(err, ShouldBeNil)

		b2, err2 := json.Marshal(query.XboxHeartbeatFilter())
		So(err2, ShouldBeNil)
		So(string(b), ShouldContainSubstring, string(b2))
	})

	// Convey("parses valid filters", t, func() {
	// 	testFilters := append(testFilters, query.StringTermsFilter("usher.broadcaster", []string{"v", "v2"}))
	// 	reader.On("Get", testChannel, testFields, testFilters).Return(&db.ResultSet{
	// 		Total: 1,
	// 		Hits: []db.ChannelResult{
	// 			testStream,
	// 		},
	// 	}, nil).Once()

	// 	v := url.Values{}
	// 	v.Set("channel", testChannel)
	// 	v.Set("fields", strings.Join(testFields, ","))
	// 	v.Set("usher.broadcaster", "v,v2")
	// 	w := makeTestRequest("GET", router.handleGet, v, []byte{})

	// 	So(w.Code, ShouldEqual, http.StatusOK)
	// 	So(w.Body.String(), ShouldContainSubstring, testChannel)
	// 	So(w.Body.String(), ShouldContainSubstring, testField)
	// })

}
