package main

import (
	"flag"
	"time"

	"code.justin.tv/common/gometrics"
	"code.justin.tv/foundation/twitchserver"

	"code.justin.tv/web/jax/api/frontend"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	_ "code.justin.tv/web/jax/common/run"
	"code.justin.tv/web/jax/common/stats"
	"code.justin.tv/web/jax/db"
)

const (
	httpReadTimeout = 10 * time.Second
	// WriteTimeout should be >30s to allow cpu profile downloads
	httpWriteTimeout = 45 * time.Second
)

var (
	confPath = "jax.json"
	addr     = ":6062"
)

func main() {
	flag.StringVar(&confPath, "conf", confPath, "Path to configuration file")
	flag.StringVar(&addr, "addr", addr, "listen address")
	flag.Parse()

	conf, err := config.New(confPath)
	log.Init(conf)
	log.SetPrefix("api: ")
	if err != nil {
		log.Printf("Error loading config: %s", err.Error())
		conf, err = config.NewFromEnvironment()
		if err != nil {
			log.Fatalf(err.Error())
		}
		log.Init(conf)
		log.SetPrefix("api: ")
	}

	log.Printf("Config: %+v", *conf)

	reader := db.NewElasticSearchReader(conf, true)
	writer := db.NewElasticSearchWriter(conf)

	statter := stats.InitStatsd(conf)
	gometrics.Monitor(statter, time.Second*5)

	fe := frontend.New(conf, reader, writer)

	err = twitchserver.ListenAndServe(fe, &twitchserver.ServerConfig{
		Addr:    addr,
		Statter: statter,
	})
	if err != nil {
		log.Fatalf("%v", err)
	}
}
