package jax

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"
	"testing"

	"code.justin.tv/foundation/twitchclient"

	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

const response = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			},
			"usher":{
				"avc_level":"3",
				"avc_profile":"Baseline",
				"average_fps":49.9323149855,
				"broadcast_part":1,
				"broadcaster":"obs",
				"channel_count":175888,
				"client_id":null,
				"configuration":"720",
				"delay_length":0,
				"hls":true,
				"id":16306092720,
				"last_updated":"2015-09-11T14:02:37.953757504-07:00",
				"max_height":226,
				"name":"live_user_esl_csgo",
				"stream_up_timestamp":1441991041,
				"up_time":"2015-09-11T17:04:01Z",
				"updated_on":"2015-09-11T21:02:30Z",
				"video_bitrate":3000.984375,
				"video_codec":"",
				"video_height":720,
				"video_width":1280
			}
		}
	}]
}`

const watchPartyResponse = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			},
			"usher":{
				"avc_level":"3",
				"avc_profile":"Baseline",
				"average_fps":49.9323149855,
				"broadcast_part":1,
				"broadcaster":"watch_party",
				"channel_count":175888,
				"client_id":null,
				"configuration":"720",
				"delay_length":0,
				"hls":true,
				"id":16306092720,
				"last_updated":"2015-09-11T14:02:37.953757504-07:00",
				"max_height":226,
				"name":"live_user_esl_csgo",
				"stream_up_timestamp":1441991041,
				"up_time":"2015-09-11T17:04:01Z",
				"updated_on":"2015-09-11T21:02:30Z",
				"video_bitrate":3000.984375,
				"video_codec":"",
				"video_height":720,
				"video_width":1280
			}
		}
	}]
}`

const watchPartyPremiereResponse = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			},
			"usher":{
				"avc_level":"3",
				"avc_profile":"Baseline",
				"average_fps":49.9323149855,
				"broadcast_part":1,
				"broadcaster":"watch_party_premiere",
				"channel_count":175888,
				"client_id":null,
				"configuration":"720",
				"delay_length":0,
				"hls":true,
				"id":16306092720,
				"last_updated":"2015-09-11T14:02:37.953757504-07:00",
				"max_height":226,
				"name":"live_user_esl_csgo",
				"stream_up_timestamp":1441991041,
				"up_time":"2015-09-11T17:04:01Z",
				"updated_on":"2015-09-11T21:02:30Z",
				"video_bitrate":3000.984375,
				"video_codec":"",
				"video_height":720,
				"video_width":1280
			}
		}
	}]
}`

const watchPartyRerunResponse = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			},
			"usher":{
				"avc_level":"3",
				"avc_profile":"Baseline",
				"average_fps":49.9323149855,
				"broadcast_part":1,
				"broadcaster":"watch_party_rerun",
				"channel_count":175888,
				"client_id":null,
				"configuration":"720",
				"delay_length":0,
				"hls":true,
				"id":16306092720,
				"last_updated":"2015-09-11T14:02:37.953757504-07:00",
				"max_height":226,
				"name":"live_user_esl_csgo",
				"stream_up_timestamp":1441991041,
				"up_time":"2015-09-11T17:04:01Z",
				"updated_on":"2015-09-11T21:02:30Z",
				"video_bitrate":3000.984375,
				"video_codec":"",
				"video_height":720,
				"video_width":1280
			}
		}
	}]
}`

const mobileBroadcastResponse = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			},
			"usher":{
				"avc_level":"3",
				"avc_profile":"Baseline",
				"average_fps":49.9323149855,
				"broadcast_part":1,
				"broadcaster":"ios",
				"channel_count":175888,
				"client_id":null,
				"configuration":"720",
				"delay_length":0,
				"hls":true,
				"id":16306092720,
				"last_updated":"2015-09-11T14:02:37.953757504-07:00",
				"max_height":226,
				"name":"live_user_esl_csgo",
				"stream_up_timestamp":1441991041,
				"up_time":"2015-09-11T17:04:01Z",
				"updated_on":"2015-09-11T21:02:30Z",
				"video_bitrate":3000.984375,
				"video_codec":"",
				"video_height":720,
				"video_width":1280
			}
		}
	}]
}`

const partneredResponse = `{
	"_total":1337,
	"hits":[{
		"channel":"esl_csgo",
		"properties":{
			"partnerships": {
				"partner_program": true
			},
			"csgo":{
				"last_updated":"2015-09-11T14:01:40.896057639-07:00",
				"map":"de_cache",
				"map_img":"http://cloud-4.steamusercontent.com/ugc/921273757974084165/4B11951A0CF99376C092E1F80C00DE719346783A/200x112.resizedimage",
				"map_name":"Cache",
				"spectators":0
			},
			"rails":{
				"broadcaster_language":"en",
				"category":"gaming",
				"channel":"esl_csgo",
				"channel_id":31239503,
				"channel_view_count":108187074,
				"directory_hidden":false,
				"game_id":32399,
				"language":"en",
				"last_updated":"2015-09-11T14:01:55.129601878-07:00",
				"meta_game":"Counter-Strike: Global Offensive",
				"steam_id":"76561198070090759",
				"title":"LIVE: ESL ESEA Pro League - Dubai Invitational",
				"xbox_gamertag":null,
				"xbox_title_id":null
			},
			"streamlist":{
				"last_updated":"2015-09-11T14:02:44.153552715-07:00"
			}
		}
	}]
}`

const summaryResponse = `{
    "results": [
        {
            "channels": 296,
            "viewers": 6410
        }
    ]
}`

const summaryGroupByGameResponse = `{
    "results": [
        {
            "channels": 296,
            "rails.game_id": 1,
            "viewers": 6410
        }
    ]
}`

const summaryGroupByCommunitySortResponse = `{
    "results": [
        {
            "channels": 296,
            "zuma.community_ids": "abc",
            "viewers": 6410,
            "sort_value": 1234
        }
    ]
}`

func TestStreamSummary(t *testing.T) {
	Convey("calling GetStreamSummary with invalid pagination", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{
			Pagination: PaginationOptions{
				Offset: -1,
			},
		}

		_, err = j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldNotBeNil)
	})

	Convey("calling GetStreamSummary", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{}

		summary, err := j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(len(summary.Results), ShouldEqual, 1)
		So(summary.Total, ShouldEqual, 1)
		So(summary.Results[0].Channels, ShouldEqual, 296)
		So(summary.Results[0].Viewers, ShouldEqual, 6410)
	})

	Convey("calling GetStreamSummary", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{}

		summary, err := j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(len(summary.Results), ShouldEqual, 1)
		So(summary.Total, ShouldEqual, 1)
		So(summary.Results[0].Channels, ShouldEqual, 296)
		So(summary.Results[0].Viewers, ShouldEqual, 6410)
	})

	Convey("calling GetStreamSummary with offset", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{
			Pagination: PaginationOptions{
				Offset: 10,
			},
		}

		summary, err := j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(len(summary.Results), ShouldEqual, 0)
		So(summary.Total, ShouldEqual, 1)
	})

	Convey("calling GetStreamSummary with group by", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryGroupByGameResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{
			GroupBy: "rails.game_id",
		}

		summary, err := j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(len(summary.Results), ShouldEqual, 1)
		So(summary.Total, ShouldEqual, 1)
		So(summary.Results[0].GroupBy, ShouldEqual, 1)
		So(summary.Results[0].SortValue, ShouldEqual, nil)
	})

	Convey("calling GetStreamSummary with group and sort", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream/summary" {
				fmt.Fprintln(w, summaryGroupByCommunitySortResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamSummaryOptions{
			GroupBy:          "zuma.community_ids",
			Sort:             "zuma.weighted_viewer_count",
			IncludeSortValue: true,
		}

		summary, err := j.GetStreamSummary(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(len(summary.Results), ShouldEqual, 1)
		So(summary.Total, ShouldEqual, 1)
		So(summary.Results[0].GroupBy, ShouldEqual, "abc")
		So(*summary.Results[0].SortValue, ShouldEqual, 1234)
	})
}

func TestStreams(t *testing.T) {
	Convey("when calling GetStreams failue", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "test=swag" {
				w.WriteHeader(http.StatusInternalServerError)
				fmt.Fprintln(w, response)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		_, err = j.GetStreams(context.Background(), r, nil)

		So(err, ShouldNotBeNil)
		So(fmt.Sprintf("%v", err), ShouldContainSubstring, strconv.Itoa(http.StatusInternalServerError))
	})

	Convey("when calling GetStreams", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "test=swag" {
				fmt.Fprintln(w, response)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Usher.StreamType), ShouldEqual, "live")
		So(*(streams.Hits[0].Properties.Usher.BroadcastPlatform), ShouldEqual, "live")
	})

	Convey("when calling GetStreams with partner", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "test=swag" {
				fmt.Fprintln(w, partneredResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Partnerships.PartnerProgram), ShouldBeTrue)
	})

	Convey("when calling GetStreams for a watch party", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "test=swag" {
				fmt.Fprintln(w, watchPartyResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Usher.StreamType), ShouldEqual, "watch_party")
		So(*(streams.Hits[0].Properties.Usher.BroadcastPlatform), ShouldEqual, "watch_party")
	})

	Convey("when calling GetStreams for a watch party premiere", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "broadcast_platform=premiere" {
				fmt.Fprintln(w, watchPartyPremiereResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("broadcast_platform", "premiere"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Usher.StreamType), ShouldEqual, "premiere")
		So(*(streams.Hits[0].Properties.Usher.BroadcastPlatform), ShouldEqual, "premiere")
	})

	Convey("when calling GetStreams for a watch party rerun", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "broadcast_platform=rerun" {
				fmt.Fprintln(w, watchPartyRerunResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("broadcast_platform", "rerun"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Usher.StreamType), ShouldEqual, "rerun")
		So(*(streams.Hits[0].Properties.Usher.BroadcastPlatform), ShouldEqual, "rerun")
	})

	Convey("when calling GetStreams for a mobile broadcast", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/streams" && r.URL.RawQuery == "test=swag" {
				fmt.Fprintln(w, mobileBroadcastResponse)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreams(context.Background(), r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
		So(*(streams.Hits[0].Properties.Usher.StreamType), ShouldEqual, "live")
		So(*(streams.Hits[0].Properties.Usher.BroadcastPlatform), ShouldEqual, "mobile")
	})
}

func TestStream(t *testing.T) {
	Convey("when calling GetStream", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream" && r.URL.RawQuery == "channel=test&test=swag" {
				fmt.Fprintln(w, response)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStream(context.Background(), "test", r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
	})
}

func TestStreamByID(t *testing.T) {
	Convey("when calling GetStream", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "GET" && r.URL.Path == "/stream" && r.URL.RawQuery == "id=test&test=swag" {
				fmt.Fprintln(w, response)
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreamByID(context.Background(), "test", r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
	})
}

func TestBulkStreams(t *testing.T) {
	Convey("when calling bulk get methods with empty lists", t, func() {
		j, err := NewClient(twitchclient.ClientConf{Host: "whatever"})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{},
		}

		streams, err := j.GetStreamsByChannels(context.Background(), []string{}, r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 0)
		So(len(streams.Hits), ShouldEqual, 0)

		streams, err = j.GetStreamsByChannelIDs(context.Background(), []string{}, r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 0)
		So(len(streams.Hits), ShouldEqual, 0)
	})

	Convey("when calling GetStreamsByChannels", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "POST" && r.URL.Path == "/stream/list" {
				r.ParseForm()
				if r.Form.Get("test") == "swag" && r.Form.Get("channels") == "c1,c2" {
					fmt.Fprintln(w, response)
				}
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreamsByChannels(context.Background(), []string{"c1", "c2"}, r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
	})

	Convey("when calling GetStreamsByChannelIDs", t, func() {
		ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			if r.Method == "POST" && r.URL.Path == "/stream/list" {
				r.ParseForm()
				if r.Form.Get("test") == "swag" && r.Form.Get("channel_ids") == "c1,c2" {
					fmt.Fprintln(w, response)
				}
			}
		}))
		defer ts.Close()

		j, err := NewClient(twitchclient.ClientConf{Host: ts.URL})
		if err != nil {
			t.Fatal(err)
		}

		r := &StreamsOptions{
			Filters: []Filter{
				EqualityFilter("test", "swag"),
			},
		}

		streams, err := j.GetStreamsByChannelIDs(context.Background(), []string{"c1", "c2"}, r, nil)

		So(err, ShouldBeNil)
		So(streams.TotalCount, ShouldEqual, 1337)
		So(len(streams.Hits), ShouldEqual, 1)
		So(streams.Hits[0].Channel, ShouldEqual, "esl_csgo")
	})
}
