package config

import (
	"log"
	"os"

	"code.justin.tv/chat/jsonconf"
)

const (
	defaultDiscoveryHost    = "discovery-production.prod.us-west2.twitch.tv:9292"
	defaultUsherHost        = "localhost:12390"
	defaultRailsHost        = "api.internal.twitch.tv:80"
	defaultChannelsHost     = "channels-prod.us-west2.twitch.tv:80"
	defaultRiotHost         = "%s.api.riotgames.com"
	defaultKalistaHost      = "kalista-production.us-west-2.elasticbeanstalk.com"
	defaultStatsHost        = "graphite.internal.justin.tv:8125"
	defaultVideoHost        = "localhost:12390"
	defaultSteamHost        = "partner.steam-api.com"
	defaultZumaHost         = "zuma.staging.us-west2.twitch.tv"
	defaultDeepMetadataHost = "vca-api.internal.twitch.tv"
	defaultMoneypennyHost   = "https://moneypenny.internal.justin.tv"
)

type Config struct {
	Environment string `json:"environment"`

	HeartbeatServiceToken string `json:"heartbeat_service_token"`
	SteamAPIKey           string `json:"steam_api_key"`
	RiotAPIToken          string `json:"lol_api_key"`
	RollbarToken          string `json:"rollbar_token"`

	ChannelsHost     string `json:"channels_host"`
	DiscoveryHost    string `json:"discovery_host"`
	KalistaHost      string `json:"kalista_host"`
	RailsHost        string `json:"rails_host"`
	RiotHost         string `json:"riot_host"`
	SteamHost        string `json:"steam_host"`
	StatsHostPort    string `json:"stats_hostport"`
	UsherHost        string `json:"usher_host"`
	ZumaHost         string `json:"zuma_host"`
	DeepMetadataHost string `json:"deepmetadata_host"`
	MoneypennyHost   string `json:"moneypenny_host"`

	ESHosts          []string `json:"es_hosts"`
	ESPort           int      `json:"es_port"`
	ESReplicateHosts []string `json:"es_replicate_hosts"`
	ESReplicateRate  float64  `json:"es_replicate_rate"`
	ESShards         int      `json:"es_shards"`
	ESReplicas       int      `json:"es_replicas"`
	ESIndex          string   `json:"es_index"`
	ESType           string   `json:"es_type"`

	KinesisIndex            string `json:"kinesis_index"`
	LanguageCCUKinesisIndex string `json:"language_ccu_kinesis_index"`
	LanguageCCUKinesisRole  string `json:"language_ccu_kinesis_role"`
	KinesisLiveChannel      string `json:"kinesis_live_channel"`

	RiotRegionPlatform map[string]string `json:"riot_region_platform"`
	SteamTimeout       float64           `json:"steam_timeout"`
	RiotScheme         string            `json:"riot_scheme"`
}

func New(path string) (*Config, error) {
	config := &Config{}
	if err := jsonconf.ReadFile(config, path); err != nil {
		return nil, err
	}
	return config, nil
}

func NewFromEnvironment() (*Config, error) {
	esHosts, err := elasticHosts()
	if err != nil {
		return nil, err
	}

	esReplicateHosts, err := replicateElasticHosts()
	if err != nil {
		log.Printf(err.Error())
	}
	esShards, esReplicas := replicationSettings()

	return &Config{
		Environment: ReadEnv("ENVIRONMENT", "development"),

		HeartbeatServiceToken: "eyJlbmMiOiJBMTI4Q0JDK0hTMjU2IiwiYWxnIjoiUlNBLU9BRVAiLCJjdHkiOiJKV1QiLCJ6aXAiOiJERUYiLCJ4NXQiOiJxcEQtU2ZoOUg3NFFHOXlMN1hSZXJ5RmZvbk0ifQ.srHYUn0HsdDDHGZv2Kn0W917lSrkFcR-hUUZriAgxT6NLyq3HP11tMEdnd2QaiQ2B7iuKKm0piIYn6fEAy6P_BJktHoXM7t7yxxRs_wHjOhXSp4ac2KDgoO61uydn7l9aVBhu_hy0TWWwsnXBKWVfhYPgHyNlr2oQoiBsjy3RDNMbhClK1-tIDnr8OZRpA-YZF1HEkTKpRWRldKQ5suLrRI-r58pQAfe7QMhAGTELbn2GwQpV7c1pmq4r7XuGx1np9cS7_xNkN2Sg2y9QUmBNXfYmF0GDskKLO1g26VbthOVoZza4obRN-HProjTiVCazkxL4ekGwUrqHxlYapBcEA.RTNWgojTzomRBtSyzxzTHA.xaaWp1q8mqK8_FjatyiD403zHzXQQdRBPWMQQhBItoKuRqiT6Dx575sr-cEmnIQS1kkm5XNV1BQs_CPVuMv4Np3NOaNc3MesC8qEQWPZDrjHzR8syoD1KTAvpBIxtYBk6b0Qjlzxklk0YdO1XizyjtmJ2SzfK6yfYXSAAce9m-6ge6dwanl81pqIV_sNPecCz09hR1uA5soYpuKMY7d-CaFd52hG03xpeCP1wMSW8QKW5ZVLUZg3wLC49MenE9ylcifKRVoKjWPr7Yek1StE51WmXFI7o8H9zs4Xk7yaXZ8AW0PemtIFexvOvMG-gyUpGddFfRn2z8jZnGmP35gSsdIIV9o9NwLP4lqyaTmif4VEs06Q6jYIpu-RjCnxBS7PD51Qok5g43J9EIwmG85iu4oUz9wYEyW7ZcfqIbIffVz31iFyrpY_ytp1la3CiNW06g75GzOk7KC18e0IHpiYLFCpnVrzwYhUzZrdqoSSQLaDe9fnPSmdx6FF7BbHkrcnCrDX9jaXpB9lFhBjWjs3TTcxmLcRY--s9N-Jm2xAKm6kVrq9qSAI2EjuPuMSR5GA0d-U5mEn_Do1eZ5E34otqJLJT3I0m1QmW-2UkMSQph1zrtnX72GbPhYZdylG0LqWJUCYFoWakEVH-u8SYrtz6LSEYjHp1KCcpPXHZEGMm8dYB7paA1CfEVWuDYqMiS-EbUUpNZPQNQDs39rup4NchCEMHoGxhnP9gkxS1Rcbj4WQlNZlohnW5gMcxQELO2BDH1jdDikzWA4eNlKE651Efc4EzJEw8TgfyJBZwcODzOxHMaLQXpmtTY1uzTCaPhM2iRhJBO79L8bE7xBGDxx147EFvn1yusJskvuM8pTrEXEu_V5IvYN78tZe-jScQmWv_AppbR8yabCCrQuMDGY_tXyNmEMWVtgCiXTG1XVdwXWd0IuJO6Bm0DwFmAzQlKZ8.a8xjpV0WnnerZBarQZqF2y2nQUBmGcRrVDlHf2jGMag",
		RiotAPIToken:          "RGAPI-0C32A0FB-542D-47C6-BEAA-32EBD2B448A1",
		RollbarToken:          "2ddbe3cebfbd419bb87845551fed1ae5",
		SteamAPIKey:           "88896D960BFB55B3156C3C20907B0F6D",

		ChannelsHost:     ReadEnv("CHANNELS_HOST", defaultChannelsHost),
		DiscoveryHost:    ReadEnv("DISCOVERY_HOST", defaultDiscoveryHost),
		RiotHost:         defaultRiotHost,
		RailsHost:        ReadEnv("RAILS_HOST", defaultRailsHost),
		StatsHostPort:    ReadEnv("STATSD_HOSTPORT", defaultStatsHost),
		SteamHost:        defaultSteamHost,
		UsherHost:        ReadEnv("USHER_HOST", defaultUsherHost),
		ZumaHost:         ReadEnv("ZUMA_HOST", defaultZumaHost),
		DeepMetadataHost: defaultDeepMetadataHost,
		MoneypennyHost:   defaultMoneypennyHost,

		ESHosts:          esHosts,
		ESPort:           elasticPort(),
		ESReplicateHosts: esReplicateHosts,
		ESReplicateRate:  replicationRate(),
		ESShards:         esShards,
		ESReplicas:       esReplicas,
		ESIndex:          "jax",
		ESType:           "channel",

		KinesisIndex:            os.Getenv("KINESIS_INDEX"),
		LanguageCCUKinesisIndex: os.Getenv("LANGUAGE_CCU_KINESIS_INDEX"),
		LanguageCCUKinesisRole:  os.Getenv("LANGUAGE_CCU_KINESIS_ROLE"),
		KinesisLiveChannel:      os.Getenv("KINESIS_LIVE_CHANNEL"),

		KalistaHost: defaultKalistaHost,
		RiotRegionPlatform: map[string]string{
			"br":   "BR1",
			"eune": "EUN1",
			"euw":  "EUW1",
			"jp":   "JP1",
			"kr":   "KR",
			"lan":  "LA1",
			"las":  "LA2",
			"na":   "NA1",
			"oce":  "OC1",
			"tr":   "TR1",
			"ru":   "RU",
			"pbe":  "PBE1",
		},
		SteamTimeout: 1.0,
		RiotScheme:   "https",
	}, nil
}
