package config

import (
	"fmt"
	"os"
	"strconv"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
)

func elasticHosts() ([]string, error) {
	if esHostsStr := os.Getenv("ES_HOSTS"); len(esHostsStr) > 0 {
		return strings.Split(esHostsStr, ","), nil
	}
	return nil, fmt.Errorf("$ES_HOSTS not found")
}

func legacyElasticHosts() ([]string, error) {
	if esHostsStr := os.Getenv("ES_LEGACY_HOSTS"); len(esHostsStr) > 0 {
		return strings.Split(esHostsStr, ","), nil
	}
	return nil, fmt.Errorf("$ES_LEGACY_HOSTS not found")
}

func replicationSettings() (int, int) {
	shards, _ := strconv.Atoi(ReadEnv("ES_SHARDS", "1"))
	replicas, _ := strconv.Atoi(ReadEnv("ES_REPLICAS", "1"))
	return shards, replicas
}

func elasticPort() int {
	if esPortStr := os.Getenv("ES_PORT"); len(esPortStr) > 0 {
		esPort, _ := strconv.Atoi(esPortStr)
		return esPort
	}
	return 9200
}

func replicateElasticHosts() ([]string, error) {
	if esHostsStr := os.Getenv("ES_REPLICATE_HOSTS"); len(esHostsStr) > 0 {
		return strings.Split(esHostsStr, ","), nil
	}
	return nil, fmt.Errorf("$ES_REPLICATE_HOSTS not found")
}

func replicationRate() float64 {
	if esReplicateRate := os.Getenv("ES_REPLICATE_RATE"); len(esReplicateRate) > 0 {
		rate, _ := strconv.ParseFloat(esReplicateRate, 64)
		return rate
	}
	return 0
}

func ReadEnv(envKey, defaultVal string) string {
	env := os.Getenv(envKey)
	if env == "" {
		env = defaultVal
	}
	return env
}

func SearchKinesisCredentials(conf *Config) *credentials.Credentials {
	awsAccessKey := ReadEnv("AWS_ACCESS_KEY_ID", "")
	awsSecretKey := ReadEnv("AWS_SECRET_ACCESS_KEY", "")

	if len(awsAccessKey) == 0 || len(awsSecretKey) == 0 {
		return credentials.NewCredentials(
			&ec2rolecreds.EC2RoleProvider{
				ExpiryWindow: 5 * time.Minute,
				Client:       ec2metadata.New(session.New(aws.NewConfig())),
			})
	}
	return credentials.NewEnvCredentials()
}

func LanguageCCUKinesisCredentials(conf *Config) *credentials.Credentials {
	kinesisRole := conf.LanguageCCUKinesisRole

	if len(kinesisRole) > 0 {
		return credentials.NewCredentials(
			&stscreds.AssumeRoleProvider{
				ExpiryWindow: 10 * time.Second,
				RoleARN:      kinesisRole,
				Client:       sts.New(session.New(aws.NewConfig())),
			})
	}

	return SearchKinesisCredentials(conf)
}
