package jsonable

import (
	"testing"

	"encoding/json"
	"time"
)

func TestRawStringUnmarshal(t *testing.T) {
	m := make(map[string]RawString)
	err := json.Unmarshal([]byte(`{"key":null}`), &m)
	if err != nil {
		t.Fatalf("unmarshal error: %v", err)
	}
	if string(m["key"]) != `null` {
		t.Errorf("incorrect unmarshal ... contents: %v", m)
	}

}

type nullBoolCase struct {
	obj       NullBool
	marshaled string
}

func TestNullBoolMarshal(t *testing.T) {
	for _, test := range []nullBoolCase{
		nullBoolCase{
			obj:       NullBool{Valid: false, Bool: false},
			marshaled: `null`,
		},
		nullBoolCase{
			obj:       NullBool{Valid: true, Bool: false},
			marshaled: `false`,
		},
		nullBoolCase{
			obj:       NullBool{Valid: true, Bool: true},
			marshaled: `true`,
		},
	} {
		b, err := json.Marshal(test.obj)
		if err != nil {
			t.Error(err)
		}
		if string(b) != test.marshaled {
			t.Errorf("wrong response for case '%+v', got '%s'", test, string(b))
		}

		obj := NullBool{}
		err = json.Unmarshal([]byte(test.marshaled), &obj)
		if err != nil {
			t.Error(err)
		}
		if obj != test.obj {
			t.Errorf("incorrect unmarshal of '%s', got '%+v'", test.marshaled, obj)
		}
	}
}

type nullInt64Case struct {
	obj       NullInt64
	marshaled string
}

func TestNullInt64Marshal(t *testing.T) {
	for _, test := range []nullInt64Case{
		nullInt64Case{
			obj:       NullInt64{Valid: false, Int64: 0},
			marshaled: `null`,
		},
		nullInt64Case{
			obj:       NullInt64{Valid: true, Int64: 0},
			marshaled: `0`,
		},
		nullInt64Case{
			obj:       NullInt64{Valid: true, Int64: 8821},
			marshaled: `8821`,
		},
	} {
		b, err := json.Marshal(test.obj)
		if err != nil {
			t.Error(err)
		}
		if string(b) != test.marshaled {
			t.Errorf("wrong response for case '%+v', got '%s'", test, string(b))
		}

		obj := NullInt64{}
		err = json.Unmarshal([]byte(test.marshaled), &obj)
		if err != nil {
			t.Error(err)
		}
		if obj != test.obj {
			t.Errorf("incorrect unmarshal of '%s', got '%+v'", test.marshaled, obj)
		}
	}
}

func TestInt64Marshal(t *testing.T) {
	var err error
	var b []byte
	var i Int64

	i = 5
	b, err = json.Marshal(i)
	if err != nil {
		t.Error(err)
	}
	if string(b) != `5` {
		t.Errorf("wrong marshal, got %#v", string(b))
	}

	b = []byte(`7`)
	err = json.Unmarshal(b, &i)
	if err != nil {
		t.Error(err)
	}
	if i != 7 {
		t.Errorf("wrong unmarshal, got %v", i)
	}

	b = []byte(`"8"`)
	err = json.Unmarshal(b, &i)
	if err != nil {
		t.Error(err)
	}
	if i != 8 {
		t.Errorf("wrong unmarshal, got %v", i)
	}
}

type nullStringCase struct {
	obj       NullString
	marshaled string
}

func TestNullStringMarshal(t *testing.T) {
	for _, test := range []nullStringCase{
		nullStringCase{
			obj:       NullString{Valid: false, String: ""},
			marshaled: `null`,
		},
		nullStringCase{
			obj:       NullString{Valid: true, String: ""},
			marshaled: `""`,
		},
		nullStringCase{
			obj:       NullString{Valid: true, String: "Kappa"},
			marshaled: `"Kappa"`,
		},
	} {
		b, err := json.Marshal(test.obj)
		if err != nil {
			t.Error(err)
		}
		if string(b) != test.marshaled {
			t.Errorf("wrong response for case '%+v', got '%s'", test, string(b))
		}

		obj := NullString{}
		err = json.Unmarshal([]byte(test.marshaled), &obj)
		if err != nil {
			t.Error(err)
		}
		if obj != test.obj {
			t.Errorf("incorrect unmarshal of '%s', got '%+v'", test.marshaled, obj)
		}
	}
}

type timeCase struct {
	obj       time.Time
	marshaled string
}

func TestTimeMarshal(t *testing.T) {
	for _, test := range []timeCase{
		timeCase{
			obj:       time.Time(time.Unix(0, 0).In(time.UTC)),
			marshaled: `"1970-01-01T00:00:00Z"`,
		},
		timeCase{
			obj:       time.Time(time.Unix(1136243045, 0).Add(-8 * time.Hour).In(time.UTC)),
			marshaled: `"2006-01-02T15:04:05Z"`,
		},
	} {
		b, err := json.Marshal(test.obj)
		if err != nil {
			t.Error(err)
		}
		if string(b) != test.marshaled {
			t.Errorf("wrong response for case '%+v', got '%s'", test, string(b))
		}

		obj := Time{}
		err = json.Unmarshal([]byte(test.marshaled), &obj)
		if err != nil {
			t.Error(err)
		}
		if !time.Time(test.obj).Equal(time.Time(obj)) {
			t.Errorf("incorrect unmarshal of '%s'", test.marshaled)
			t.Errorf("have %s", time.Time(obj).Format(time.RFC1123Z))
			t.Errorf("want %s", time.Time(test.obj).Format(time.RFC1123Z))
		}
	}
}

func BenchmarkNullBoolMarshal(b *testing.B) {
	// BenchmarkNullBool_Marshal	  500000	      3134 ns/op
	n := NullBool{Valid: false, Bool: false}
	t := NullBool{Valid: true, Bool: true}
	f := NullBool{Valid: true, Bool: false}
	for i := 0; i < b.N; i++ {
		_, _ = json.Marshal(&n)
		_, _ = json.Marshal(&t)
		_, _ = json.Marshal(&f)
	}
}

func BenchmarkNullInt64Marshal(b *testing.B) {
	// BenchmarkNullInt64_Marshal	  500000	      3198 ns/op
	n := NullInt64{Valid: false, Int64: 0}
	t := NullInt64{Valid: true, Int64: 5231423}
	f := NullInt64{Valid: true, Int64: 0}
	for i := 0; i < b.N; i++ {
		_, _ = json.Marshal(&n)
		_, _ = json.Marshal(&t)
		_, _ = json.Marshal(&f)
	}
}

func BenchmarkNullStringMarshal(b *testing.B) {
	// BenchmarkNullString_Marshal	  500000	      4049 ns/op
	n := NullString{Valid: false, String: ""}
	t := NullString{Valid: true, String: "Kappa"}
	f := NullString{Valid: true, String: ""}
	for i := 0; i < b.N; i++ {
		_, _ = json.Marshal(&n)
		_, _ = json.Marshal(&t)
		_, _ = json.Marshal(&f)
	}
}

func BenchmarkNullBoolUnmarshal(b *testing.B) {
	// BenchmarkNullBool_Unmarshal	  500000	      7237 ns/op
	n := []byte(`null`)
	t := []byte(`true`)
	f := []byte(`false`)
	obj := NullBool{}
	for i := 0; i < b.N; i++ {
		_ = json.Unmarshal(n, &obj)
		_ = json.Unmarshal(t, &obj)
		_ = json.Unmarshal(f, &obj)
	}
}

func BenchmarkNullInt64Unmarshal(b *testing.B) {
	// BenchmarkNullInt64_Unmarshal	  500000	      6061 ns/op
	n := []byte(`null`)
	t := []byte(`12318467424284`)
	f := []byte(`0`)
	obj := NullInt64{}
	for i := 0; i < b.N; i++ {
		_ = json.Unmarshal(n, &obj)
		_ = json.Unmarshal(t, &obj)
		_ = json.Unmarshal(f, &obj)
	}
}

func BenchmarkNullStringUnmarshal(b *testing.B) {
	// BenchmarkNullString_Unmarshal	  200000	      7624 ns/op
	n := []byte(`null`)
	t := []byte(`"Kappa"`)
	f := []byte(`""`)
	obj := NullString{}
	for i := 0; i < b.N; i++ {
		_ = json.Unmarshal(n, &obj)
		_ = json.Unmarshal(t, &obj)
		_ = json.Unmarshal(f, &obj)
	}
}

func BenchmarkStarBoolMarshal(b *testing.B) {
	// BenchmarkStarBool_Marshal	 1000000	      2206 ns/op
	// there's a small performance improvement here, but really not enough to be worth it
	var n, t, f *bool
	eltT, eltF := true, false
	t, f = &eltT, &eltF
	for i := 0; i < b.N; i++ {
		_, _ = json.Marshal(&n)
		_, _ = json.Marshal(&t)
		_, _ = json.Marshal(&f)
	}
}
