package jsonable

import (
	"database/sql"
	"encoding/json"
	"fmt"
)

// NullBool is a nullable bool type.
type NullBool sql.NullBool

var _ json.Unmarshaler = new(NullBool)
var _ json.Marshaler = *new(NullBool)

// UnmarshalJSON takes a JSON string describing a NullBool and
// initializes this NullBool instance.
func (T *NullBool) UnmarshalJSON(b []byte) (err error) {
	var in interface{}
	if err = json.Unmarshal(b, &in); err != nil {
		return
	}
	switch v := in.(type) {
	// there's no "case nil:" ... this code never gets the nil state
	// a json `null` is interpreted as the zero value for this type,
	// and this method isn't called
	case bool:
		T.Valid, T.Bool = true, v
	case string:
		switch v {
		case "true", "1":
			T.Valid, T.Bool = true, true
		case "false", "0":
			T.Valid, T.Bool = true, false
		default:
			// TODO: maybe we're interested in logging these?
		}
	default:
		// TODO: maybe we're interested in logging these?
	}
	return
}

// MarshalJSON takes this NullBool and returns it as JSON in b.
func (T NullBool) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b = []byte(fmt.Sprintf("%t", T.Bool))
	}
	return
}

// AsNullString returns this NullBool as a NullString.
func (T NullBool) AsNullString() (out sql.NullString) {
	return sql.NullString{
		Valid:  T.Valid,
		String: fmt.Sprintf("%t", T.Bool),
	}
}
