package jsonable

import (
	"bytes"
	"database/sql"
	"encoding/json"
	"fmt"
	"strconv"
)

// NullInt64 is a nullable int64.
type NullInt64 sql.NullInt64

var _ json.Unmarshaler = new(NullInt64)
var _ json.Marshaler = *new(NullInt64)

// UnmarshalJSON takes the JSON-encoded input and unpacks it into this
// NullInt64.
func (T *NullInt64) UnmarshalJSON(b []byte) (err error) {
	if bytes.Equal(b, []byte(`null`)) {
		T.Valid, T.Int64 = false, 0
		return
	}
	if err = json.Unmarshal(b, &T.Int64); err != nil {
		if typeErr, ok := err.(*json.UnmarshalTypeError); ok && typeErr.Value == "null" {
			// we tried to unpack `null` into an int64. it didn't work.
			// this is fine! we'll just null out our value.
			T.Valid, T.Int64 = false, 0
			err = nil
		}
		return
	}
	T.Valid = true
	return
}

// MarshalJSON takes this NullInt64 and packs it in b, JSON-encoded.
func (T NullInt64) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b = []byte(fmt.Sprintf("%d", T.Int64))
	}
	return
}

// AsNullString returns this NullInt64 as a NullString.
func (T NullInt64) AsNullString() (out sql.NullString) {
	return sql.NullString{
		Valid:  T.Valid,
		String: fmt.Sprintf("%d", T.Int64),
	}
}

// Int64 formally implements MarshalJSON/UnmarshalJSON for the int64 type
type Int64 int64

var _ json.Unmarshaler = new(Int64)
var _ json.Marshaler = *new(Int64)

// UnmarshalJSON takes the JSON-encoded input and puts it into this Int64.
func (T *Int64) UnmarshalJSON(b []byte) error {
	var interr error
	var out int64
	interr = json.Unmarshal(b, &out)
	if typeErr, ok := interr.(*json.UnmarshalTypeError); ok && typeErr.Value == "string" {
		// maybe we tried to unmarshal a stringed int?
		var s string
		var strerr error
		if strerr = json.Unmarshal(b, &s); strerr != nil {
			// it's not a string either ... return the original error
			return interr
		}
		if out, strerr = strconv.ParseInt(s, 10, 64); strerr != nil {
			return strerr
		}
		interr = nil
	}
	*T = Int64(out)
	return interr
}

// MarshalJSON takes this Int64 and packs it into b, JSON-encoded.
func (T Int64) MarshalJSON() (b []byte, err error) {
	b = strconv.AppendInt(nil, int64(T), 10)
	return
}
