package jsonable

import (
	"database/sql"
	"encoding/json"
)

// NullString is a nullable string.
type NullString sql.NullString

var _ json.Unmarshaler = new(NullString)
var _ json.Marshaler = *new(NullString)

// UnmarshalJSON takes the JSON-encoded input and puts it into this NullString.
func (T *NullString) UnmarshalJSON(b []byte) (err error) {
	var in interface{}
	if err = json.Unmarshal(b, &in); err != nil {
		return
	}
	switch v := in.(type) {
	case string:
		T.Valid, T.String = true, v
	case nil:
		T.Valid, T.String = false, ""
	default:
		// TODO: maybe we're interested in logging these?
	}
	return
}

// MarshalJSON takes this NullString and puts it into b, JSON-encoded.
func (T NullString) MarshalJSON() (b []byte, err error) {
	b = []byte(`null`)
	if T.Valid {
		b, err = json.Marshal(&T.String)
	}
	return
}

// AsNullString returns this NullString as a sql.NullString
func (T NullString) AsNullString() (out sql.NullString) {
	return sql.NullString(T)
}

// String formally implements MarshalJSON for the string type
type String string

var _ json.Marshaler = *new(String)

// MarshalJSON takes this String and puts it into b, JSON-encoded.
func (T String) MarshalJSON() (b []byte, err error) {
	return json.Marshal(string(T))
}
