package log

import (
	"fmt"
	"log"
	"net/http"
	"os"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/common/golibs/errorlogger/rollbar"
	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/stats"
)

const (
	numWorkers = 10
)

var (
	// DefaultLogger is the standard, STDOUT logger.
	DefaultLogger *log.Logger

	// Stats is the method of sending stats to statsd.
	Stats statsd.Statter

	// ErrorLogger is an alternative loging system.
	ErrorLogger errorlogger.ErrorLogger
)

func Init(conf *config.Config) {
	DefaultLogger = log.New(os.Stdout, "", 0)

	if conf != nil {
		SetStatsLogger(stats.InitStatsd(conf))
		ErrorLogger = rollbar.NewRollbarLogger(conf.RollbarToken, conf.Environment)
	}
}

// SetPrefix sets a prefix so different process can log to the same file and
// still be seperable.
func SetPrefix(prefix string) {
	DefaultLogger.SetPrefix(prefix)
}

// SetStatsLogger sets the stats logger for this process.
func SetStatsLogger(l statsd.Statter) {
	Stats = l
}

// ReportStats sends the event name and status to the stats server.
// Also logs how much time passed since the start of this request and now.
func ReportStats(name string, start time.Time, status int) {
	if Stats != nil {
		Stats.TimingDuration(name, time.Now().Sub(start), 0.1)
		Stats.Inc(fmt.Sprintf("response.%s.%d", name, status), 1, 0.1)
	}
}

func RequestError(r *http.Request, status int, err error) {
	ErrorLogger.RequestError(r, err)
}

// Reportf logs a string and sends it to Rollbar if specified. Usually
// used for information which is good to know, but not a big problem.
func Reportf(format string, v ...interface{}) {
	errstring := fmt.Sprintf(format, v...)
	if ErrorLogger != nil {
		ErrorLogger.Error(fmt.Errorf(format, v...))
	}
	DefaultLogger.Printf(errstring)
}

// Printf simply logs the string to the DefaultLogger.
func Printf(format string, v ...interface{}) {
	DefaultLogger.Printf(format, v...)
}

// Fatalf logs a string and sends it to Rollbar, then exists the
// process. This is for major errors.
func Fatalf(format string, v ...interface{}) {
	errstring := fmt.Sprintf(format, v...)
	if ErrorLogger != nil {
		ErrorLogger.Error(fmt.Errorf("Fatal error: %s", errstring))
	}

	DefaultLogger.Fatal(errstring)
}

// loggingResponseWriter captures the status code and timing of an http.Handler.
type loggingResponseWriter struct {
	http.ResponseWriter
	name  string
	start time.Time
}

// NewLoggingHandler wraps a handler function, passing it a loggingResponseWriter.
func NewLoggingHandler(name string, handler func(http.ResponseWriter, *http.Request)) func(http.ResponseWriter, *http.Request) {
	return func(w http.ResponseWriter, r *http.Request) {
		handler(NewLoggingResponseWriter(w, name), r)
	}
}

// NewLoggingResponseWriter wraps an http.ResponseWriter with a loggingResponserWriter.
func NewLoggingResponseWriter(w http.ResponseWriter, name string) loggingResponseWriter {
	return loggingResponseWriter{
		ResponseWriter: w,
		name:           name,
		start:          time.Now(),
	}
}

// WriteHeader reports stats about the executed requests and writes the response header.
func (T loggingResponseWriter) WriteHeader(status int) {
	ReportStats(T.name, T.start, status)
	T.ResponseWriter.WriteHeader(status)
}
