package backend

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/log"
	. "code.justin.tv/web/jax/common/lol/models"
)

const (
	summonerQuery     = "/summonername"
	summonerRankQuery = "/lol/league/v3/leagues/by-summoner/%s"
	currentMatchQuery = "/lol/spectator/v3/active-games/by-summoner/%s"
	kalistaScheme     = "http"
)

type Backend interface {
	FetchSummonerName(twitchID string) TwitchSummoner
	FetchSummonerRank(platform string, summonerID int64) SummonerRank
	FetchCurrentMatch(platform string, summonerID int64) Match
}

type backendImpl struct {
	Config *config.Config
}

var _ Backend = new(backendImpl)

func New(conf *config.Config) (Backend, error) {
	return &backendImpl{
		Config: conf,
	}, nil
}

func (T *backendImpl) makeKalistaAPIRequest(path string, v url.Values) (resp io.Reader, err error) {
	u := &url.URL{
		Scheme:   kalistaScheme,
		Host:     T.Config.KalistaHost,
		Path:     path,
		RawQuery: v.Encode(),
	}

	r, err := http.Get(u.String())
	if err != nil {
		fmt.Printf("error: bad GET request %v\n", err)
		return
	}
	resp = r.Body
	if r.StatusCode != http.StatusOK {
		fmt.Printf("error: bad status from kalista %d\n", r.StatusCode)
	}
	return
}

func (T *backendImpl) FetchSummonerName(twitchID string) TwitchSummoner {
	v := url.Values{
		"twitchid": {twitchID},
	}
	resp, err := T.makeKalistaAPIRequest(summonerQuery, v)
	if err != nil {
		return TwitchSummoner{}
	}

	var response []TwitchSummoner
	decoder := json.NewDecoder(resp)
	if err = decoder.Decode(&response); err != nil {
		fmt.Printf("error: cannot decode response for summoner names (%+v)\n", err)
	}

	if len(response) == 0 {
		return TwitchSummoner{}
	}

	return response[0]
}

func (T *backendImpl) makeRiotAPIRequest(path string, v url.Values, platform string) (resp io.Reader, err error) {
	u := &url.URL{
		Scheme:   T.Config.RiotScheme,
		Host:     fmt.Sprintf(T.Config.RiotHost, platform),
		Path:     path,
		RawQuery: v.Encode(),
	}

	r, err := http.Get(u.String())
	if err != nil {
		fmt.Printf("error: bad GET request %v\n", err)
		return
	}
	resp = r.Body
	if r.StatusCode != http.StatusOK && r.StatusCode != http.StatusNotFound {
		fmt.Printf("error: bad status from riot %d: %v\n", r.StatusCode, u.String())
	}
	return
}

func (T *backendImpl) FetchSummonerRank(platform string, summonerID int64) SummonerRank {
	v := url.Values{
		"api_key": {T.Config.RiotAPIToken},
	}

	log.Stats.Inc("riot.api.rank", 1, 1)
	resp, err := T.makeRiotAPIRequest(fmt.Sprintf(summonerRankQuery, strconv.FormatInt(summonerID, 10)), v, platform)
	if err != nil {
		return SummonerRank{}
	}
	var response []SummonerRank
	decoder := json.NewDecoder(resp)
	if err = decoder.Decode(&response); err != nil {
		fmt.Printf("error: cannot decode response for summoner rank %+v\n", err)
		return SummonerRank{}
	}

	if len(response) == 0 {
		return SummonerRank{}
	}

	return response[0]
}

func (T *backendImpl) FetchCurrentMatch(platform string, summonerID int64) Match {
	v := url.Values{
		"api_key": {T.Config.RiotAPIToken},
	}
	log.Stats.Inc("riot.api.match", 1, 1)
	resp, err := T.makeRiotAPIRequest(fmt.Sprintf(currentMatchQuery, strconv.FormatInt(summonerID, 10)), v, platform)
	if err != nil {
		return Match{}
	}

	var match Match
	decoder := json.NewDecoder(resp)
	if err = decoder.Decode(&match); err != nil {
		fmt.Printf("error: cannot decode response for match %+v\n", err)
	}

	return match
}
