package logic

import (
	"fmt"

	"code.justin.tv/web/jax/common/config"
	"code.justin.tv/web/jax/common/lol/backend"
	. "code.justin.tv/web/jax/common/lol/models"
)

const (
	soloQueue = "RANKED_SOLO_5x5"
)

type Logic interface {
	GetLoLData(twitchID string) (LoLProperties, error)
	GetLoLRank(twitchID string) (LoLRankProperties, error)
}

type logicImpl struct {
	backend backend.Backend
	config  *config.Config
}

var _ Logic = new(logicImpl)

// New creates a new Logic struct given a Backend struct
func New(conf *config.Config, b backend.Backend) (Logic, error) {
	return &logicImpl{
		backend: b,
		config:  conf,
	}, nil
}

func (T *logicImpl) GetLoLData(twitchID string) (LoLProperties, error) {
	props := LoLProperties{}

	summoner := T.backend.FetchSummonerName(twitchID)
	if summoner.SummonerID == 0 {
		return props, fmt.Errorf("summoner not found for %v", twitchID)
	}

	props.TwitchID = twitchID
	props.SummonerID = summoner.SummonerID
	props.SummonerName = summoner.SummonerName
	props.Region = summoner.Region
	props.IsConnected = true

	match := T.backend.FetchCurrentMatch(T.regionToPlatformID(props.Region), props.SummonerID)
	for _, participant := range match.Participants {
		if participant.SummonerID == props.SummonerID {
			props.ChampionID = participant.ChampionID
		}
	}

	props.GameStartTime = match.GameStartTime / 1000

	return props, nil
}

func (T *logicImpl) GetLoLRank(twitchID string) (LoLRankProperties, error) {
	props := LoLRankProperties{}

	summoner := T.backend.FetchSummonerName(twitchID)
	if summoner.SummonerID == 0 {
		return props, fmt.Errorf("summoner not found for %v", twitchID)
	}

	rank := T.backend.FetchSummonerRank(T.regionToPlatformID(summoner.Region), summoner.SummonerID)
	if rank.Queue == soloQueue {
		props.Rank = rank.Tier
		props.RankNumeric = T.rankToNumeric(rank.Tier)
	}

	return props, nil
}

func (T *logicImpl) regionToPlatformID(region string) string {
	platformID, ok := T.config.RiotRegionPlatform[region]
	if !ok {
		return "NA1"
	}
	return platformID
}

// rankToNumeric takes a tier and converts it to a numeric value that is then sortable by Jax
func (T *logicImpl) rankToNumeric(tier string) int {
	tierScores := map[string]int{
		"BRONZE":     100000,
		"SILVER":     200000,
		"GOLD":       300000,
		"PLATINUM":   400000,
		"DIAMOND":    500000,
		"MASTER":     600000,
		"CHALLENGER": 700000,
	}
	return tierScores[tier]
}
