package backend

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"

	"code.justin.tv/web/jax/common/config"
	. "code.justin.tv/web/jax/common/pubg/models"
)

const (
	deepmetadataQuery  = "/v1/deep_metadata/channel/%s"
	deepmetadataScheme = "http"
)

type PUBGBackend interface {
	FetchPUBGDeepMetadata(channelID string) (response PUBGDeepmetadataEvent, err error)
}

type backendImpl struct {
	Config *config.Config
}

var _ PUBGBackend = new(backendImpl)

func New(conf *config.Config) (PUBGBackend, error) {
	return &backendImpl{
		Config: conf,
	}, nil
}

func (T *backendImpl) makeDeepmetadataAPIRequest(path string, v url.Values, pathVars ...interface{}) (resp io.Reader, err error) {
	u := &url.URL{
		Scheme:   deepmetadataScheme,
		Host:     T.Config.DeepMetadataHost,
		Path:     fmt.Sprintf(path, pathVars...),
		RawQuery: v.Encode(),
	}
	r, err := http.Get(u.String())

	if err != nil {
		fmt.Printf("error: bad GET request %v\n", err)
		return
	}
	resp = r.Body
	if r.StatusCode != http.StatusOK {
		fmt.Printf("error: bad status from deepmetadata API %d\n", r.StatusCode)
	}
	return
}

func (T *backendImpl) FetchPUBGDeepMetadata(channelID string) (pubgResponse PUBGDeepmetadataEvent, err error) {
	resp, err := T.makeDeepmetadataAPIRequest(deepmetadataQuery, nil, channelID)
	if err != nil {
		return
	}
	response := PUBGStreamDeepMetadata{}
	decoder := json.NewDecoder(resp)
	if err = decoder.Decode(&response); err != nil {
		fmt.Printf("error: cannot decode response for summoner names (%+v)\n", err)
		return
	}
	pubgResponse = response.PUBGData
	return
}
