package xbox

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
)

const (
	heartbeatURL = "https://userpresence.xboxlive.com/externalbroadcasts"
)

type heartbeatRequest struct {
	headers        map[string]string
	marshalledBody []byte
}

// HeartbeatRequestBody is the request body to send to xbox presence service
type HeartbeatRequestBody struct {
	Gamertag    string `json:"gt"`
	Titleid     string
	Broadcastid string
	Started     int
	Viewers     string
}

func (X *heartbeatRequest) init(xstsToken string, marshalledBody []byte) {
	X.headers = map[string]string{
		"Authorization":          "XBL3.0 x=-;" + xstsToken,
		"X-xbl-contract-version": "1",
		"Content-Type":           "application/json",
	}
}

func makeHeartbeatRequest(headers map[string]string, body []byte) (validChs map[string]bool, statusCode int, err error) {
	client := &http.Client{}
	req, _ := http.NewRequest("POST", heartbeatURL, bytes.NewBuffer(body))
	for k, v := range headers {
		req.Header.Add(k, v)
	}
	resp, err := client.Do(req)

	if err != nil {
		fmt.Printf("ERROR %s\n", err)
		return
	}
	defer resp.Body.Close()
	statusCode = resp.StatusCode
	decoder := json.NewDecoder(resp.Body)
	var chs []string
	if err := decoder.Decode(&chs); err != nil {
		err = fmt.Errorf("Error decoding heartbeat response: %v", err)
	}

	validChs = make(map[string]bool)
	for _, ch := range chs {
		validChs[ch] = true
	}
	return
}

// MakeHeartbeatRequest updates the xbox presence service with channel metadata with the appropriate service token
func MakeHeartbeatRequest(body []HeartbeatRequestBody, xstsToken string) (validChs map[string]bool, err error) {
	marshalledBody, _ := json.Marshal(body)

	heartbeat := new(heartbeatRequest)
	heartbeat.init(xstsToken, marshalledBody)
	heartbeat.marshalledBody = marshalledBody

	validChs, statusCode, err := makeHeartbeatRequest(heartbeat.headers, marshalledBody)
	fmt.Printf("Heartbeat Request Status Code: %+v\n", statusCode)
	return
}
