package xbox

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/rand"
	"crypto/sha256"
	"encoding/base64"
	"encoding/binary"
	"fmt"
	"time"
)

const (
	windowsUnixOffset = 11644473600
)

// SigningData contains the data required to create the encrypted digest used in Microsoft APIs
type SigningData struct {
	policyVersion      uint32
	timestamp          uint64 // 64bit big endian time
	httpMethod         string
	pathAndQueryString string
	authHeader         string // should be none hopefully?
	marshalledBody     []byte
	privateKey         *ecdsa.PrivateKey
}

// convert from windows file timestamp (10 years before epoch time), network encoded
func getWinTimestamp() (winTime uint64) {
	t := uint64(time.Now().Unix())
	winTime = (t + windowsUnixOffset) * 10 * 1000 * 1000

	return
}

func padZero(buf *bytes.Buffer) {
	err := binary.Write(buf, binary.BigEndian, uint8(0))
	if err != nil {
		fmt.Println("error when adding 0 byte")
	}
}

func (signingData *SigningData) generateDigest() (timestamp uint64, digest []byte, err error) {
	buf := new(bytes.Buffer)
	timestamp = getWinTimestamp()

	// prologue
	var data = []interface{}{
		signingData.policyVersion,
		int64(timestamp),
		[]byte(signingData.httpMethod),
		[]byte(signingData.pathAndQueryString),
		[]byte(signingData.authHeader),
		signingData.marshalledBody,
	}
	for _, v := range data {
		err = binary.Write(buf, binary.BigEndian, v)
		padZero(buf)

		if err != nil {
			fmt.Printf("binary.Write failed for value %v, %v:\n", v, err)
		}
	}

	digest = buf.Bytes()
	return
}

func (signingData *SigningData) sign(data []byte) (signedData []byte, err error) {
	hashed := sha256.Sum256(data)
	r, s, err := ecdsa.Sign(rand.Reader, signingData.privateKey, hashed[:])

	b := new(bytes.Buffer)
	b.Write(r.Bytes())
	b.Write(s.Bytes())

	if err != nil {
		fmt.Println("error signing message")
	}
	signedData = b.Bytes()
	return
}

func (signingData *SigningData) signatureHeader() (encodedSignature string, err error) {
	timestamp, digest, err := signingData.generateDigest()
	if err != nil {
		fmt.Println("error generating digest")
	}

	signedData, err := signingData.sign(digest)
	if err != nil {
		fmt.Println("error signing data")
	}

	buf := new(bytes.Buffer)
	var data = []interface{}{
		signingData.policyVersion,
		int64(timestamp),
		signedData,
	}

	for _, v := range data {
		err = binary.Write(buf, binary.BigEndian, v)
		if err != nil {
			fmt.Println("binary.Write failed:", err)
		}
	}

	encodedSignature = base64.StdEncoding.EncodeToString(buf.Bytes())
	return
}
