package xbox

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

var (
	xstsToken      string
	xstsExpiration time.Time
)

const (
	xstsPolicyVersion = 1
	xstsURL           = "https://xsts.auth.xboxlive.com/xsts/authorize"
	xstsRelyingParty  = "http://xboxlive.com"
	xstsPath          = "/xsts/authorize"

	msDatetimeFormatBase = "2006-01-02T15:04:05"
)

type xstsRequest struct {
	headers      map[string]string
	extraHeaders map[string]string
	signingData  SigningData
}

func (X *xstsRequest) Init(privateKey *ecdsa.PrivateKey, marshalledBody []byte) {
	X.headers = map[string]string{
		"X-xbl-contract-version": "1",
		"Content-Type":           "application/json",
	}
	X.extraHeaders = map[string]string{}

	X.signingData = SigningData{
		policyVersion:      xstsPolicyVersion,
		httpMethod:         "POST",
		pathAndQueryString: xstsPath,
		privateKey:         privateKey,
		marshalledBody:     marshalledBody,
	}

	signatureHeader, err := X.signingData.signatureHeader()
	if err != nil {
		fmt.Println("error generating signature header")
	}
	X.headers["Signature"] = signatureHeader
}

type xstsRequestBody struct {
	RelyingParty string
	TokenType    string
	Properties   xstsPropertyBag
}

type xstsPropertyBag struct {
	ServiceToken string
}

type xstsResponse struct {
	IssueInstant string
	NotAfter     string
	Token        string
}

func makeRequest(headers map[string]string, body []byte) (respBody []byte, err error) {
	client := &http.Client{}
	req, _ := http.NewRequest("POST", xstsURL, bytes.NewBuffer(body))
	for k, v := range headers {
		req.Header.Add(k, v)
	}

	resp, err := client.Do(req)
	if err != nil {
		fmt.Printf("ERROR %s\n", err)
		return
	}
	defer resp.Body.Close()
	respBody, err = ioutil.ReadAll(resp.Body)
	return
}

func makexstsRequest(serviceToken string) (xstsResponse xstsResponse, err error) {
	privateKey, _ := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)

	body := xstsRequestBody{
		RelyingParty: xstsRelyingParty,
		TokenType:    "JWT",
		Properties: xstsPropertyBag{
			ServiceToken: serviceToken,
		},
	}
	marshalledBody, _ := json.Marshal(body)

	xsts := new(xstsRequest)
	xsts.Init(privateKey, marshalledBody)

	headers := map[string]string{}
	for k, v := range xsts.headers {
		headers[k] = v
	}

	respBody, err := makeRequest(headers, marshalledBody)
	json.Unmarshal(respBody, &xstsResponse)
	return
}

// Parses the timestamp given by MS api endpoint. Handles arbitrary decimal precision.
func parseMSTime(raw string) (t time.Time) {
	// checks the length of the decimals minus the 'Z' and handles no decimals
	trailingDecimals := ""
	if split := strings.Split(raw, "."); len(split) > 1 {
		decimalPlaces := len(strings.Split(raw, ".")[1]) - 1
		trailingDecimals = "." + strings.Repeat("0", decimalPlaces) + "Z"
	}
	t, _ = time.Parse(msDatetimeFormatBase+trailingDecimals, raw)
	return
}

// GetXToken returns the service token required to make a call into the xbox presence service
func GetXToken(serviceToken string) (xtoken string, err error) {
	now := time.Now().Add(-5 * time.Minute)
	if xstsToken == "" || xstsExpiration.Before(now) {
		resp, err := makexstsRequest(string(serviceToken))
		if err != nil {
			fmt.Println("Error making XSTS request: %v", err.Error())
		}

		xstsToken = resp.Token
		xstsExpiration = parseMSTime(resp.NotAfter)
	}

	xtoken = xstsToken
	return
}
