package db

import (
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/web/jax/db/query"
	es "github.com/Kaidence/elastigo/lib"
)

// JaxReader defines the different ways to query Jax's backend for information
type JaxReader interface {
	Get(channel string, fields []string, filters []query.Filter) (*ResultSet, *JaxDbError)
	GetByID(channel string, fields []string, filters []query.Filter) (*ResultSet, *JaxDbError)
	BulkGetByChannel(channels []string, fields []string, sortField string, limit, offset int, filters ...query.Filter) (*ResultSet, *JaxDbError)
	BulkGetByChannelID(channelIDs []string, fields []string, sortField string, limit, offset int, filters ...query.Filter) (*ResultSet, *JaxDbError)
	Search(args map[string]interface{}, q query.SearchQuery) (*ResultSet, *JaxDbError)
	Aggregate(q query.AggregationQuery) ([]Aggregate, *JaxDbError)
	Lease(statName string, bufferSize int, scanFilters ...query.Filter) chan []ChannelResult
	Scan(statName string, bufferSize int, scanFilters ...query.Filter) chan []ChannelResult
}

// JaxWriter is used to update the information stored in Jax's backend. It takes
// a channel name and a map of properties.
type JaxWriter interface {
	Update(channel string, input map[string]interface{}) *JaxDbError
	Delete(channel string)
	CreateIndexIfNotExists(shards, replicas int) *JaxDbError
}

// ChannelResult describes the search results for a single channel.
type ChannelResult struct {
	Channel    string                 `json:"channel"`
	Properties map[string]interface{} `json:"properties"`
}

func (T ChannelResult) GetID() int64 {
	if railsProps, ok := T.Properties["rails"].(map[string]interface{}); ok {
		if num, ok := railsProps["channel_id"].(json.Number); ok {
			if id, err := num.Int64(); err == nil {
				return id
			}
		}
	}
	// try to still work even when flattened
	if num, ok := T.Properties["rails.channel_id"].(json.Number); ok {
		if id, err := num.Int64(); err == nil {
			return id
		}
	}

	return -1
}

func (T ChannelResult) getViewerCount() int64 {
	if usherProps, ok := T.Properties["usher"].(map[string]interface{}); ok {
		if num, ok := usherProps["channel_count"].(json.Number); ok {
			if count, err := num.Int64(); err == nil {
				return count
			}
		}
	}

	return 0
}

func (T ChannelResult) getStreamUpTimestamp() int64 {
	if usherProps, ok := T.Properties["usher"].(map[string]interface{}); ok {
		if num, ok := usherProps["stream_up_timestamp"].(json.Number); ok {
			if count, err := num.Int64(); err == nil {
				return count
			}
		}
	}

	return 0
}

// Aggregation response from ElasticSearch
type AggregateResponse struct {
	Results struct {
		Buckets []Aggregate `json:"buckets"`
	}
}

type Aggregate struct {
	Value     interface{} `json:"value"`
	Viewers   int         `json:"viewers"`
	SortValue *int        `json:"sort_value"`
	Channels  int         `json:"channels"`
}

func (a *Aggregate) UnmarshalJSON(b []byte) (err error) {
	m := map[string]interface{}{}
	err = json.Unmarshal(b, &m)

	if err != nil {
		return
	}

	a.Channels = int(m["doc_count"].(float64))
	a.Viewers = int(m["total_count"].(map[string]interface{})["value"].(float64))
	if _, ok := m["alt_sort_count"]; ok {
		sortValue := int(m["alt_sort_count"].(map[string]interface{})["value"].(float64))
		a.SortValue = &sortValue
	}
	a.Value = m["key"]
	return
}

// ResultSet describes all the search results for a single query.
// ScrollId can be set if the results are paginated.
type ResultSet struct {
	ScrollID string          `json:"-"`
	Total    int             `json:"_total"`
	Hits     []ChannelResult `json:"hits"`
}

// JaxDbError is an implementation of error to give more information about
// backend-related errors.
type JaxDbError struct {
	Code    int
	Message string
	Time    time.Time
}

func (T *JaxDbError) Error() string {
	return fmt.Sprintf("(%v): [%v] %v", T.Time, T.Code, T.Message)
}

func NewJaxDbError(msg string) *JaxDbError {
	return &JaxDbError{Code: http.StatusInternalServerError, Message: msg,
		Time: time.Now()}
}

// formatError takes a regular error and formats it into a JaxDbError with more
// information.
func formatError(err error) *JaxDbError {
	// This probably only gets called when the error is in fact an underlying elastigo error
	if err != nil {
		switch t := err.(type) {
		case es.ESError:
			return &JaxDbError{Code: t.Code, Message: t.What, Time: t.When}
		default:
			return NewJaxDbError(err.Error())
		}
	}
	return nil
}
