package db

import (
	"encoding/json"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestGetID(t *testing.T) {
	Convey("without id", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{},
		}

		So(channel.GetID(), ShouldEqual, -1)
	})

	Convey("with malformed id", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"rails": map[string]interface{}{
					"channel_id": "23llkj4",
				},
			},
		}

		So(channel.GetID(), ShouldEqual, -1)
	})

	Convey("with id", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"rails": map[string]interface{}{
					"channel_id": json.Number("123"),
				},
			},
		}

		So(channel.GetID(), ShouldEqual, 123)
	})
}

func TestGetViewers(t *testing.T) {
	Convey("without channel_count", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{},
		}

		So(channel.getViewerCount(), ShouldEqual, 0)
	})

	Convey("with malformed channel_count", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"channel_count": "23llkj4",
				},
			},
		}

		So(channel.getViewerCount(), ShouldEqual, 0)
	})

	Convey("with channel_count", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"channel_count": json.Number("123"),
				},
			},
		}

		So(channel.getViewerCount(), ShouldEqual, 123)
	})
}

func getStreamUpTimestamp(t *testing.T) {
	Convey("without stream_up_timestamp", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{},
		}

		So(channel.getStreamUpTimestamp(), ShouldEqual, 0)
	})

	Convey("with malformed stream_up_timestamp", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"stream_up_timestamp": "23llkj4",
				},
			},
		}

		So(channel.getStreamUpTimestamp(), ShouldEqual, 0)
	})

	Convey("with stream_up_timestamp", t, func() {
		channel := ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"stream_up_timestamp": json.Number("123"),
				},
			},
		}

		So(channel.getStreamUpTimestamp(), ShouldEqual, 123)
	})
}

func TestUnmarshalAggregateResponse(t *testing.T) {
	Convey("malformed", t, func() {
		body := `{
			"what"
		}
		`
		var a AggregateResponse
		err := json.Unmarshal([]byte(body), &a)
		So(err, ShouldNotBeNil)
	})

	Convey("with sort_value", t, func() {
		body := `
			{
				"results": {
					"buckets": [
						{
							"key": "foo",
							"doc_count": 1.0,
							"total_count": {
								"value": 100.0
							},
							"alt_sort_count": {
								"value": 100.0
							}
						}
					]
				}
			}
		`
		var a AggregateResponse
		err := json.Unmarshal([]byte(body), &a)
		So(err, ShouldBeNil)
		So(len(a.Results.Buckets), ShouldEqual, 1)
		So(a.Results.Buckets[0].Channels, ShouldEqual, 1)
		So(a.Results.Buckets[0].Viewers, ShouldEqual, 100)
		So(*a.Results.Buckets[0].SortValue, ShouldEqual, 100)
	})

	Convey("without sort_value", t, func() {
		body := `
			{
				"results": {
					"buckets": [
						{
							"key": "foo",
							"doc_count": 1.0,
							"total_count": {
								"value": 100.0
							}
						}
					]
				}
			}
		`
		var a AggregateResponse
		err := json.Unmarshal([]byte(body), &a)
		So(err, ShouldBeNil)
		So(len(a.Results.Buckets), ShouldEqual, 1)
		So(a.Results.Buckets[0].Channels, ShouldEqual, 1)
		So(a.Results.Buckets[0].Viewers, ShouldEqual, 100)
		So(a.Results.Buckets[0].SortValue, ShouldEqual, nil)
	})
}
