package db

import (
	"encoding/json"
	"sort"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

var (
	jsonProperties = map[string]interface{}{
		"parent1": map[string]interface{}{
			"property1": map[string]interface{}{
				"nested1": "value1",
				"nested2": "value1",
			},
			"property2": "value2",
		},
		"parent2": map[string]interface{}{
			"property3": "value3",
		},
	}

	flatProperties = map[string]interface{}{
		"parent1.property1.nested1": "value1",
		"parent1.property1.nested2": "value1",
		"parent1.property2":         "value2",
		"parent2.property3":         "value3",
	}
)

func TestFlattenProperties(t *testing.T) {
	Convey("flattens correctly", t, func() {
		flat := flattenProperties("", jsonProperties)
		So(flat, ShouldResemble, flatProperties)
	})
}

func TestJsonifyProperties(t *testing.T) {
	Convey("jsonifies correctly", t, func() {
		json := jsonifyProperties(flatProperties, []string{})
		So(json, ShouldResemble, jsonProperties)
	})

	Convey("jsonifies correctly with a field list", t, func() {
		json := jsonifyProperties(flatProperties, []string{"parent1.property1.nested1"})
		So(json, ShouldResemble, map[string]interface{}{
			"parent1": map[string]interface{}{
				"property1": map[string]interface{}{
					"nested1": "value1",
				},
			},
		})
	})
}

func TestSortByViewers(t *testing.T) {
	channels := []ChannelResult{
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"channel_count": json.Number("1"),
				},
			},
		},
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"channel_count": json.Number("3"),
				},
			},
		},
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"channel_count": json.Number("2"),
				},
			},
		},
	}

	Convey("sorts", t, func() {
		sort.Sort(ByChannelCount(channels))
		So(channels[0].getViewerCount(), ShouldEqual, 3)
		So(channels[1].getViewerCount(), ShouldEqual, 2)
		So(channels[2].getViewerCount(), ShouldEqual, 1)
	})
}

func TestSortByRecency(t *testing.T) {
	channels := []ChannelResult{
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"stream_up_timestamp": json.Number("1"),
				},
			},
		},
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"stream_up_timestamp": json.Number("3"),
				},
			},
		},
		ChannelResult{
			Properties: map[string]interface{}{
				"usher": map[string]interface{}{
					"stream_up_timestamp": json.Number("2"),
				},
			},
		},
	}

	Convey("sorts", t, func() {
		sort.Sort(ByRecency(channels))
		So(channels[0].getStreamUpTimestamp(), ShouldEqual, 3)
		So(channels[1].getStreamUpTimestamp(), ShouldEqual, 2)
		So(channels[2].getStreamUpTimestamp(), ShouldEqual, 1)
	})
}
